<?php
defined( 'ABSPATH' ) || exit;

/**
 * Adds B2B fields (SIREN, SIRET, TVA) to WooCommerce checkout.
 */
class WCEF_Checkout_Fields {

	public static function init(): void {
		add_filter( 'woocommerce_billing_fields', array( __CLASS__, 'add_fields' ) );
		add_action( 'woocommerce_checkout_process', array( __CLASS__, 'validate' ) );
		add_action( 'woocommerce_checkout_update_order_meta', array( __CLASS__, 'save' ) );
		add_action( 'woocommerce_admin_order_data_after_billing_address', array( __CLASS__, 'display_in_admin' ) );
	}

	/**
	 * Add B2B fields to billing section.
	 */
	public static function add_fields( array $fields ): array {
		$fields['billing_is_business'] = array(
			'type'     => 'checkbox',
			'label'    => __( 'Je commande en tant que professionnel', 'wc-efacturation-france' ),
			'required' => false,
			'class'    => array( 'form-row-wide' ),
			'priority' => 25,
		);

		$fields['billing_siren'] = array(
			'type'     => 'text',
			'label'    => __( 'SIREN', 'wc-efacturation-france' ),
			'required' => false,
			'class'    => array( 'form-row-first', 'wcef-b2b-field' ),
			'priority' => 26,
			'maxlength' => 9,
		);

		$fields['billing_siret'] = array(
			'type'     => 'text',
			'label'    => __( 'SIRET', 'wc-efacturation-france' ),
			'required' => false,
			'class'    => array( 'form-row-last', 'wcef-b2b-field' ),
			'priority' => 27,
			'maxlength' => 14,
		);

		$fields['billing_tva_intra'] = array(
			'type'     => 'text',
			'label'    => __( 'N° TVA intracommunautaire', 'wc-efacturation-france' ),
			'required' => false,
			'class'    => array( 'form-row-wide', 'wcef-b2b-field' ),
			'priority' => 28,
		);

		return $fields;
	}

	/**
	 * Validate B2B fields if business checkbox is checked.
	 */
	public static function validate(): void {
		if ( empty( $_POST['billing_is_business'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
			return;
		}

		$siren = sanitize_text_field( wp_unslash( $_POST['billing_siren'] ?? '' ) ); // phpcs:ignore WordPress.Security.NonceVerification
		if ( ! empty( $siren ) && ! self::validate_siren( $siren ) ) {
			wc_add_notice( __( 'Le SIREN doit contenir exactement 9 chiffres.', 'wc-efacturation-france' ), 'error' );
		}

		$siret = sanitize_text_field( wp_unslash( $_POST['billing_siret'] ?? '' ) ); // phpcs:ignore WordPress.Security.NonceVerification
		if ( ! empty( $siret ) && ! self::validate_siret( $siret ) ) {
			wc_add_notice( __( 'Le SIRET doit contenir exactement 14 chiffres.', 'wc-efacturation-france' ), 'error' );
		}

		$tva = sanitize_text_field( wp_unslash( $_POST['billing_tva_intra'] ?? '' ) ); // phpcs:ignore WordPress.Security.NonceVerification
		if ( ! empty( $tva ) && ! self::validate_tva_intra( $tva ) ) {
			wc_add_notice( __( 'Le numero de TVA doit etre au format FR suivi de 11 chiffres.', 'wc-efacturation-france' ), 'error' );
		}
	}

	/**
	 * Save B2B fields to order meta.
	 */
	public static function save( int $order_id ): void {
		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		$fields = array( 'billing_is_business', 'billing_siren', 'billing_siret', 'billing_tva_intra' );
		foreach ( $fields as $field ) {
			$value = sanitize_text_field( wp_unslash( $_POST[ $field ] ?? '' ) ); // phpcs:ignore WordPress.Security.NonceVerification
			if ( '' !== $value ) {
				$order->update_meta_data( '_' . $field, $value );
			}
		}
		$order->save();
	}

	/**
	 * Display B2B fields in admin order view.
	 */
	public static function display_in_admin( WC_Order $order ): void {
		$is_biz = $order->get_meta( '_billing_is_business' );
		if ( ! $is_biz ) {
			return;
		}

		echo '<div class="wcef-b2b-admin">';
		echo '<h3>' . esc_html__( 'Informations B2B', 'wc-efacturation-france' ) . '</h3>';

		$siren = $order->get_meta( '_billing_siren' );
		$siret = $order->get_meta( '_billing_siret' );
		$tva   = $order->get_meta( '_billing_tva_intra' );

		if ( $siren ) {
			echo '<p><strong>SIREN :</strong> ' . esc_html( $siren ) . '</p>';
		}
		if ( $siret ) {
			echo '<p><strong>SIRET :</strong> ' . esc_html( $siret ) . '</p>';
		}
		if ( $tva ) {
			echo '<p><strong>TVA Intra :</strong> ' . esc_html( $tva ) . '</p>';
		}

		echo '</div>';
	}

	/**
	 * Validate SIREN (9 digits).
	 */
	public static function validate_siren( string $siren ): bool {
		return (bool) preg_match( '/^\d{9}$/', $siren );
	}

	/**
	 * Validate SIRET (14 digits).
	 */
	public static function validate_siret( string $siret ): bool {
		return (bool) preg_match( '/^\d{14}$/', $siret );
	}

	/**
	 * Validate French TVA intracommunautaire.
	 */
	public static function validate_tva_intra( string $tva ): bool {
		return (bool) preg_match( '/^FR\d{11}$/', strtoupper( $tva ) );
	}
}
