<?php
defined( 'ABSPATH' ) || exit;

/**
 * Chorus Pro API connector via PISTE.
 *
 * The plugin acts as an OD (Operateur de Dematerialisation) — no PDP certification required.
 * The user provides their own PISTE/Chorus Pro credentials.
 */
class WCEF_Chorus_Pro {

	const ENV_SANDBOX    = 'sandbox';
	const ENV_PRODUCTION = 'production';

	private static array $endpoints = array(
		'sandbox'    => array(
			'api'   => 'https://sandbox-api.piste.gouv.fr/cpro',
			'oauth' => 'https://sandbox-oauth.piste.gouv.fr',
		),
		'production' => array(
			'api'   => 'https://api.piste.gouv.fr/cpro',
			'oauth' => 'https://oauth.piste.gouv.fr',
		),
	);

	/**
	 * Get the currently configured environment.
	 */
	public static function get_environment(): string {
		return get_option( 'wcef_chorus_environment', self::ENV_SANDBOX );
	}

	/**
	 * Check if Chorus Pro is configured with credentials.
	 */
	public static function is_configured(): bool {
		return ! empty( get_option( 'wcef_chorus_client_id', '' ) )
			&& ! empty( get_option( 'wcef_chorus_client_secret', '' ) )
			&& ! empty( get_option( 'wcef_chorus_tech_login', '' ) )
			&& ! empty( get_option( 'wcef_chorus_tech_password', '' ) );
	}

	/**
	 * Get an OAuth2 access token (cached in transient).
	 */
	private static function get_access_token(): string {
		$env       = self::get_environment();
		$cache_key = 'wcef_chorus_token_' . $env;
		$cached    = get_transient( $cache_key );
		if ( $cached ) {
			return $cached;
		}

		$oauth_url     = self::$endpoints[ $env ]['oauth'];
		$client_id     = get_option( 'wcef_chorus_client_id', '' );
		$client_secret = get_option( 'wcef_chorus_client_secret', '' );

		$response = wp_remote_post( $oauth_url . '/api/oauth/token', array(
			'timeout' => 30,
			'headers' => array( 'Content-Type' => 'application/x-www-form-urlencoded' ),
			'body'    => http_build_query( array(
				'grant_type'    => 'client_credentials',
				'client_id'     => $client_id,
				'client_secret' => $client_secret,
				'scope'         => 'openid',
			) ),
		) );

		if ( is_wp_error( $response ) ) {
			return '';
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );
		if ( empty( $body['access_token'] ) ) {
			return '';
		}

		$ttl = ( $body['expires_in'] ?? 3600 ) - 60;
		set_transient( $cache_key, $body['access_token'], max( 60, $ttl ) );
		return $body['access_token'];
	}

	/**
	 * Build request headers for all API calls.
	 *
	 * @return array|WP_Error
	 */
	private static function get_headers() {
		$token = self::get_access_token();
		if ( empty( $token ) ) {
			return new WP_Error( 'oauth_error', __( 'Impossible d\'obtenir un token OAuth. Verifiez vos identifiants PISTE.', 'wc-efacturation-france' ) );
		}

		$login    = get_option( 'wcef_chorus_tech_login', '' );
		$password = get_option( 'wcef_chorus_tech_password', '' );
		// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
		$cpro_account = base64_encode( $login . ':' . $password );

		return array(
			'Authorization' => 'Bearer ' . $token,
			'cpro-account'  => $cpro_account,
			'Content-Type'  => 'application/json',
			'Accept'        => 'application/json',
		);
	}

	/**
	 * Make a POST API call.
	 *
	 * @return array|WP_Error
	 */
	private static function api_post( string $path, array $body ) {
		$headers = self::get_headers();
		if ( is_wp_error( $headers ) ) {
			return $headers;
		}

		$env     = self::get_environment();
		$api_url = self::$endpoints[ $env ]['api'] . $path;

		$response = wp_remote_post( $api_url, array(
			'timeout' => 60,
			'headers' => $headers,
			'body'    => wp_json_encode( $body ),
		) );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$code = wp_remote_retrieve_response_code( $response );
		$data = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( $code >= 400 ) {
			$error_msg = $data['message'] ?? $data['libelle'] ?? 'Erreur API Chorus Pro (HTTP ' . $code . ')';
			return new WP_Error( 'chorus_api_error', $error_msg, array( 'status' => $code, 'body' => $data ) );
		}

		return $data ?: array();
	}

	/**
	 * Submit an invoice to Chorus Pro.
	 *
	 * @param object $invoice Invoice row from DB.
	 * @return array|WP_Error Chorus Pro response or error.
	 */
	public static function submit_invoice( object $invoice ) {
		if ( ! WCEF_License::can( 'routage_pa' ) ) {
			return new WP_Error( 'feature_gated', __( 'Votre plan ne permet pas le routage PA.', 'wc-efacturation-france' ) );
		}

		if ( ! self::is_configured() ) {
			return new WP_Error( 'not_configured', __( 'Chorus Pro n\'est pas configure. Allez dans Reglages > Chorus Pro.', 'wc-efacturation-france' ) );
		}

		if ( ! WCEF_Transmission_Counter::can_transmit() ) {
			return new WP_Error( 'limit_reached', __( 'Limite mensuelle de transmissions atteinte.', 'wc-efacturation-france' ) );
		}

		$order = wc_get_order( $invoice->order_id );
		if ( ! $order ) {
			return new WP_Error( 'invalid_order', __( 'Commande introuvable.', 'wc-efacturation-france' ) );
		}

		// Set status to pending before sending.
		WCEF_Invoice::update_status( $invoice->id, 'pending_transmission', __( 'Envoi en cours vers Chorus Pro.', 'wc-efacturation-france' ) );

		// Build payload and submit.
		$payload = self::build_submission_payload( $invoice, $order );
		$result  = self::api_post( '/factures/v1/soumettre', $payload );

		if ( is_wp_error( $result ) ) {
			WCEF_Invoice::update_status(
				$invoice->id,
				'error',
				__( 'Erreur de transmission : ', 'wc-efacturation-france' ) . $result->get_error_message()
			);
			global $wpdb;
			$wpdb->update(
				$wpdb->prefix . 'wcef_invoices',
				array( 'transmission_error' => $result->get_error_message() ),
				array( 'id' => $invoice->id ),
				array( '%s' ),
				array( '%d' )
			);
			return $result;
		}

		// Store Chorus Pro identifiers.
		global $wpdb;
		$chorus_id = sanitize_text_field( $result['identifiantFactureCPP'] ?? '' );
		$wpdb->update(
			$wpdb->prefix . 'wcef_invoices',
			array(
				'chorus_id'          => $chorus_id,
				'transmitted_at'     => current_time( 'mysql' ),
				'transmission_error' => '',
			),
			array( 'id' => $invoice->id ),
			array( '%s', '%s', '%s' ),
			array( '%d' )
		);

		WCEF_Invoice::update_status(
			$invoice->id,
			'transmitted',
			sprintf( __( 'Facture transmise a Chorus Pro. ID: %s', 'wc-efacturation-france' ), $chorus_id ?: 'N/A' ),
			wp_json_encode( $result )
		);

		return $result;
	}

	/**
	 * Build the JSON payload for invoice submission.
	 */
	private static function build_submission_payload( object $invoice, WC_Order $order ): array {
		$structure_id = get_option( 'wcef_chorus_structure_id', '' );
		$buyer_siret  = $order->get_meta( '_billing_siret' ) ?: '';

		$line_items = array();
		$line_num   = 1;
		foreach ( $order->get_items() as $item ) {
			$product  = $item->get_product();
			$line_ht  = (float) $item->get_total();
			$line_tax = (float) $item->get_total_tax();

			$line_items[] = array(
				'numeroLignePoste'   => $line_num,
				'referencePoste'     => $product ? $product->get_sku() : '',
				'designationPoste'   => $item->get_name(),
				'quantitePoste'      => $item->get_quantity(),
				'unitePoste'         => 'unite',
				'prixUnitairePoste'  => round( $line_ht / max( 1, $item->get_quantity() ), 5 ),
				'montantHtPoste'     => round( $line_ht, 2 ),
				'tauxTvaPoste'       => 20,
				'montantTvaPoste'    => round( $line_tax, 2 ),
				'montantTtcPoste'    => round( $line_ht + $line_tax, 2 ),
			);
			$line_num++;
		}

		return array(
			'cadreDeFacturation'  => array(
				'codeCadreFacturation' => 'A1_FACTURE_FOURNISSEUR',
			),
			'identifiantDestinataire' => array(
				'codeDestinataire' => $buyer_siret,
				'typeDestinataire' => 'SIRET',
			),
			'identifiantFournisseur' => array(
				'idFournisseur'   => $structure_id,
				'typeFournisseur' => 'STRUCTURE',
			),
			'references' => array(
				'deviseFacture'     => $order->get_currency(),
				'modeDepot'         => 'SAISIE_API',
				'numeroFacture'     => $invoice->invoice_number,
				'typeFacture'       => 'FACTURE',
				'typeTva'           => 'TVA_SUR_DEBIT',
			),
			'lignesPoste' => $line_items,
			'pieceJointeComplementaire' => array(
				array(
					'pieceJointeDesignation' => 'facture-' . $invoice->invoice_number . '.xml',
					'pieceJointeType'        => 'text/xml',
					// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
					'pieceJointeContenu'     => base64_encode( $invoice->xml_content ),
				),
			),
		);
	}

	/**
	 * Check the status of a submitted invoice on Chorus Pro.
	 *
	 * @return array|WP_Error
	 */
	public static function check_status( object $invoice ) {
		if ( empty( $invoice->chorus_id ) ) {
			return new WP_Error( 'no_chorus_id', __( 'Cette facture n\'a pas ete transmise a Chorus Pro.', 'wc-efacturation-france' ) );
		}

		$result = self::api_post( '/factures/v1/consulter', array(
			'identifiantFactureCPP' => $invoice->chorus_id,
		) );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		$chorus_status = $result['statutFacture'] ?? '';
		$mapped_status = self::map_chorus_status( $chorus_status );

		if ( $mapped_status && $mapped_status !== $invoice->status ) {
			WCEF_Invoice::update_status(
				$invoice->id,
				$mapped_status,
				sprintf( __( 'Statut mis a jour depuis Chorus Pro: %s', 'wc-efacturation-france' ), $chorus_status ),
				wp_json_encode( $result )
			);
		}

		return $result;
	}

	/**
	 * Map Chorus Pro status to plugin status.
	 */
	private static function map_chorus_status( string $chorus_status ): string {
		$map = array(
			'DEPOSEE'             => 'transmitted',
			'EN_COURS_TRAITEMENT' => 'transmitted',
			'MISE_A_DISPOSITION'  => 'delivered',
			'ACCEPTEE'            => 'accepted',
			'REJETEE'             => 'rejected',
			'REFUSEE'             => 'rejected',
			'MANDATEE'            => 'delivered',
			'MISE_EN_PAIEMENT'    => 'delivered',
			'SUSPENDUE'           => 'transmitted',
		);
		return $map[ $chorus_status ] ?? '';
	}

	/**
	 * Search invoices received as a recipient (supplier invoices).
	 *
	 * @return array|WP_Error
	 */
	public static function fetch_received_invoices( array $params = array() ) {
		if ( ! WCEF_License::can( 'reception_fournisseurs' ) ) {
			return new WP_Error( 'feature_gated', __( 'Cette fonctionnalite necessite un plan superieur.', 'wc-efacturation-france' ) );
		}

		$structure_id = get_option( 'wcef_chorus_structure_id', '' );
		$body = array_merge( array(
			'identifiantStructure' => $structure_id,
			'typeRecepteur'        => 'STRUCTURE',
			'nbResultatsParPage'   => 50,
			'pageResultatDemandee' => 1,
		), $params );

		return self::api_post( '/factures/v1/rechercher/recipiendaire', $body );
	}

	/**
	 * Test the connection with current credentials.
	 *
	 * @return array|WP_Error
	 */
	public static function test_connection() {
		if ( ! self::is_configured() ) {
			return new WP_Error( 'not_configured', __( 'Identifiants manquants.', 'wc-efacturation-france' ) );
		}

		$token = self::get_access_token();
		if ( empty( $token ) ) {
			return new WP_Error( 'oauth_error', __( 'Echec de l\'authentification OAuth. Verifiez Client ID et Client Secret.', 'wc-efacturation-france' ) );
		}

		// Use the rattachements endpoint to verify both OAuth + cpro-account work.
		return self::api_post( '/transverses/v1/recuperer/rattachementsComptesTechniques', array(
			'idStructure' => get_option( 'wcef_chorus_structure_id', '' ),
		) );
	}
}
