<?php
defined( 'ABSPATH' ) || exit;

/**
 * WP-Cron jobs for Chorus Pro status polling and supplier invoice sync.
 */
class WCEF_Cron {

	const HOOK_STATUS_CHECK  = 'wcef_check_chorus_statuses';
	const HOOK_SUPPLIER_SYNC = 'wcef_sync_supplier_invoices';

	/**
	 * Register cron action hooks.
	 */
	public static function init(): void {
		add_action( self::HOOK_STATUS_CHECK, array( __CLASS__, 'check_statuses' ) );
		add_action( self::HOOK_SUPPLIER_SYNC, array( __CLASS__, 'sync_suppliers' ) );
	}

	/**
	 * Schedule cron events if not already scheduled.
	 */
	public static function schedule(): void {
		if ( ! wp_next_scheduled( self::HOOK_STATUS_CHECK ) ) {
			wp_schedule_event( time(), 'hourly', self::HOOK_STATUS_CHECK );
		}
		if ( ! wp_next_scheduled( self::HOOK_SUPPLIER_SYNC ) ) {
			wp_schedule_event( time(), 'twicedaily', self::HOOK_SUPPLIER_SYNC );
		}
	}

	/**
	 * Clear all scheduled cron events.
	 */
	public static function unschedule(): void {
		wp_clear_scheduled_hook( self::HOOK_STATUS_CHECK );
		wp_clear_scheduled_hook( self::HOOK_SUPPLIER_SYNC );
	}

	/**
	 * Check statuses for all transmitted invoices that are not yet in a final state.
	 */
	public static function check_statuses(): void {
		if ( ! WCEF_License::can( 'statuts_temps_reel' ) ) {
			return;
		}
		if ( ! WCEF_Chorus_Pro::is_configured() ) {
			return;
		}

		global $wpdb;
		$table    = $wpdb->prefix . 'wcef_invoices';
		$invoices = $wpdb->get_results(
			"SELECT * FROM {$table}
			 WHERE chorus_id != ''
			 AND status IN ('pending_transmission', 'transmitted')
			 ORDER BY transmitted_at ASC
			 LIMIT 50" // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		);

		foreach ( $invoices as $invoice ) {
			WCEF_Chorus_Pro::check_status( $invoice );
			usleep( 200000 ); // 200ms delay to avoid rate limiting.
		}
	}

	/**
	 * Sync supplier invoices from Chorus Pro.
	 */
	public static function sync_suppliers(): void {
		if ( ! WCEF_License::can( 'reception_fournisseurs' ) ) {
			return;
		}
		if ( ! WCEF_Chorus_Pro::is_configured() ) {
			return;
		}

		WCEF_Supplier_Invoice::sync();
	}
}
