<?php
defined( 'ABSPATH' ) || exit;

/**
 * Handles plugin activation, deactivation, and DB setup.
 */
class WCEF_Install {

	const DB_VERSION = '1.1.0';

	/**
	 * Run on activation.
	 */
	public static function activate(): void {
		self::create_tables();
		update_option( 'wcef_version', WCEF_VERSION );
		update_option( 'wcef_installed_at', time() );
	}

	/**
	 * Run on deactivation.
	 */
	public static function deactivate(): void {
		WCEF_Cron::unschedule();
	}

	/**
	 * Check if DB needs upgrading and run migrations.
	 */
	public static function maybe_upgrade(): void {
		$current = get_option( 'wcef_db_version', '1.0.0' );
		if ( version_compare( $current, self::DB_VERSION, '>=' ) ) {
			return;
		}
		self::create_tables();
		update_option( 'wcef_db_version', self::DB_VERSION );
	}

	/**
	 * Create/update all custom database tables via dbDelta.
	 */
	private static function create_tables(): void {
		global $wpdb;

		$charset = $wpdb->get_charset_collate();

		// Main invoices table (with new transmission columns).
		$invoices = $wpdb->prefix . 'wcef_invoices';
		$sql_invoices = "CREATE TABLE {$invoices} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			order_id bigint(20) unsigned NOT NULL,
			invoice_number varchar(50) NOT NULL,
			invoice_type varchar(10) NOT NULL DEFAULT '380',
			status varchar(30) NOT NULL DEFAULT 'generated',
			buyer_siren varchar(14) DEFAULT '',
			buyer_tva varchar(20) DEFAULT '',
			total_ht decimal(12,2) NOT NULL DEFAULT 0,
			total_tax decimal(12,2) NOT NULL DEFAULT 0,
			total_ttc decimal(12,2) NOT NULL DEFAULT 0,
			xml_content longtext DEFAULT NULL,
			pdf_path varchar(500) DEFAULT '',
			error_message text DEFAULT NULL,
			chorus_id varchar(100) DEFAULT '',
			transmitted_at datetime DEFAULT NULL,
			status_updated_at datetime DEFAULT NULL,
			transmission_error text DEFAULT NULL,
			created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY  (id),
			UNIQUE KEY invoice_number (invoice_number),
			KEY order_id (order_id),
			KEY status (status),
			KEY chorus_id (chorus_id)
		) {$charset};";

		// Transmission log table (status timeline).
		$log = $wpdb->prefix . 'wcef_transmission_log';
		$sql_log = "CREATE TABLE {$log} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			invoice_id bigint(20) unsigned NOT NULL,
			status varchar(30) NOT NULL,
			message text DEFAULT NULL,
			chorus_response text DEFAULT NULL,
			created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY  (id),
			KEY invoice_id (invoice_id),
			KEY created_at (created_at)
		) {$charset};";

		// Supplier invoices table (incoming from Chorus Pro).
		$suppliers = $wpdb->prefix . 'wcef_supplier_invoices';
		$sql_suppliers = "CREATE TABLE {$suppliers} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			chorus_id varchar(100) NOT NULL DEFAULT '',
			supplier_name varchar(255) DEFAULT '',
			supplier_siren varchar(14) DEFAULT '',
			supplier_siret varchar(14) DEFAULT '',
			invoice_number varchar(100) NOT NULL DEFAULT '',
			invoice_date date DEFAULT NULL,
			total_ht decimal(12,2) NOT NULL DEFAULT 0,
			total_tax decimal(12,2) NOT NULL DEFAULT 0,
			total_ttc decimal(12,2) NOT NULL DEFAULT 0,
			status varchar(30) NOT NULL DEFAULT 'received',
			xml_content longtext DEFAULT NULL,
			pdf_path varchar(500) DEFAULT '',
			received_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY  (id),
			UNIQUE KEY chorus_id (chorus_id),
			KEY supplier_siren (supplier_siren),
			KEY status (status)
		) {$charset};";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta( $sql_invoices );
		dbDelta( $sql_log );
		dbDelta( $sql_suppliers );
	}
}
