<?php
defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Invoices list table for the admin page.
 */
class WCEF_Invoice_List extends WP_List_Table {

	public function __construct() {
		parent::__construct( array(
			'singular' => 'invoice',
			'plural'   => 'invoices',
			'ajax'     => false,
		) );
	}

	public function get_columns(): array {
		$columns = array();

		// Add checkbox column for bulk actions when routing is available.
		if ( WCEF_License::can( 'routage_pa' ) ) {
			$columns['cb'] = '<input type="checkbox" />';
		}

		$columns += array(
			'invoice_number' => __( 'Facture', 'wc-efacturation-france' ),
			'order_id'       => __( 'Commande', 'wc-efacturation-france' ),
			'buyer'          => __( 'Client', 'wc-efacturation-france' ),
			'total_ht'       => __( 'Total HT', 'wc-efacturation-france' ),
			'total_ttc'      => __( 'Total TTC', 'wc-efacturation-france' ),
			'status'         => __( 'Statut', 'wc-efacturation-france' ),
			'created_at'     => __( 'Date', 'wc-efacturation-france' ),
			'actions'        => __( 'Actions', 'wc-efacturation-france' ),
		);

		return $columns;
	}

	public function column_cb( $item ): string {
		return '<input type="checkbox" name="invoice_ids[]" value="' . esc_attr( $item->id ) . '">';
	}

	public function get_bulk_actions(): array {
		$actions = array();
		if ( WCEF_License::can( 'routage_pa' ) ) {
			$actions['transmit'] = __( 'Transmettre a la PA', 'wc-efacturation-france' );
		}
		return $actions;
	}

	public function get_sortable_columns(): array {
		return array(
			'invoice_number' => array( 'invoice_number', false ),
			'total_ttc'      => array( 'total_ttc', false ),
			'created_at'     => array( 'created_at', true ),
		);
	}

	public function prepare_items(): void {
		$per_page = 20;
		$page     = $this->get_pagenum();

		$args = array(
			'per_page' => $per_page,
			'offset'   => ( $page - 1 ) * $per_page,
			'orderby'  => sanitize_text_field( wp_unslash( $_GET['orderby'] ?? 'created_at' ) ), // phpcs:ignore WordPress.Security.NonceVerification
			'order'    => sanitize_text_field( wp_unslash( $_GET['order'] ?? 'DESC' ) ), // phpcs:ignore WordPress.Security.NonceVerification
			'status'   => sanitize_text_field( wp_unslash( $_GET['status'] ?? '' ) ), // phpcs:ignore WordPress.Security.NonceVerification
			'search'   => sanitize_text_field( wp_unslash( $_GET['s'] ?? '' ) ), // phpcs:ignore WordPress.Security.NonceVerification
		);

		$result = WCEF_Invoice::list( $args );

		$this->items = $result['items'];
		$this->set_pagination_args( array(
			'total_items' => $result['total'],
			'per_page'    => $per_page,
			'total_pages' => ceil( $result['total'] / $per_page ),
		) );

		$this->_column_headers = array(
			$this->get_columns(),
			array(),
			$this->get_sortable_columns(),
		);
	}

	public function column_default( $item, $column_name ): string {
		return esc_html( $item->$column_name ?? '' );
	}

	public function column_invoice_number( object $item ): string {
		return '<strong>' . esc_html( $item->invoice_number ) . '</strong>';
	}

	public function column_order_id( object $item ): string {
		$order = wc_get_order( $item->order_id );
		if ( ! $order ) {
			return '#' . esc_html( $item->order_id );
		}
		$url = $order->get_edit_order_url();
		return '<a href="' . esc_url( $url ) . '">#' . esc_html( $order->get_order_number() ) . '</a>';
	}

	public function column_buyer( object $item ): string {
		$order = wc_get_order( $item->order_id );
		if ( ! $order ) {
			return '-';
		}
		$name = $order->get_billing_company() ?: ( $order->get_billing_first_name() . ' ' . $order->get_billing_last_name() );
		$out  = esc_html( $name );
		if ( $item->buyer_siren ) {
			$out .= '<br><small>SIREN: ' . esc_html( $item->buyer_siren ) . '</small>';
		}
		return $out;
	}

	public function column_total_ht( object $item ): string {
		return esc_html( number_format( (float) $item->total_ht, 2, ',', ' ' ) ) . ' &euro;';
	}

	public function column_total_ttc( object $item ): string {
		return '<strong>' . esc_html( number_format( (float) $item->total_ttc, 2, ',', ' ' ) ) . ' &euro;</strong>';
	}

	public function column_status( object $item ): string {
		$labels = WCEF_Invoice::get_status_labels();
		$label  = $labels[ $item->status ] ?? $item->status;

		return '<span class="wcef-badge wcef-badge-' . esc_attr( $item->status ) . '">'
			. esc_html( $label )
			. '</span>';
	}

	public function column_created_at( object $item ): string {
		return esc_html( date_i18n( 'd/m/Y H:i', strtotime( $item->created_at ) ) );
	}

	public function column_actions( object $item ): string {
		$pdf_url = wp_nonce_url(
			admin_url( 'admin-ajax.php?action=wcef_download_pdf&invoice_id=' . absint( $item->id ) ),
			'wcef_nonce',
			'nonce'
		);
		$xml_url = wp_nonce_url(
			admin_url( 'admin-ajax.php?action=wcef_download_xml&invoice_id=' . absint( $item->id ) ),
			'wcef_nonce',
			'nonce'
		);

		$html = '<div class="wcef-actions">'
			. '<a href="' . esc_url( $pdf_url ) . '" class="button wcef-btn-pdf" target="_blank">PDF</a>'
			. '<a href="' . esc_url( $xml_url ) . '" class="button wcef-btn-xml">XML</a>';

		// Transmit button — only for generated invoices when routing is available.
		if ( WCEF_License::can( 'routage_pa' ) && 'generated' === $item->status ) {
			$html .= '<button type="button" class="button wcef-btn-transmit" data-invoice-id="'
				. esc_attr( $item->id ) . '">'
				. esc_html__( 'Transmettre', 'wc-efacturation-france' ) . '</button>';
		}

		// Refresh status button — for transmitted invoices.
		if ( WCEF_License::can( 'statuts_temps_reel' ) && in_array( $item->status, array( 'pending_transmission', 'transmitted' ), true ) ) {
			$html .= '<button type="button" class="button wcef-btn-refresh-status" data-invoice-id="'
				. esc_attr( $item->id ) . '" title="'
				. esc_attr__( 'Rafraichir le statut', 'wc-efacturation-france' ) . '">&#8635;</button>';
		}

		$html .= '</div>';
		return $html;
	}

	/**
	 * Extra table nav (status filter).
	 */
	protected function extra_tablenav( $which ): void {
		if ( 'top' !== $which ) {
			return;
		}

		$current = sanitize_text_field( wp_unslash( $_GET['status'] ?? '' ) ); // phpcs:ignore WordPress.Security.NonceVerification
		$all_statuses = WCEF_Invoice::get_status_labels();
		?>
		<div class="alignleft actions">
			<select name="status">
				<option value=""><?php esc_html_e( 'Tous les statuts', 'wc-efacturation-france' ); ?></option>
				<?php foreach ( $all_statuses as $value => $label ) : ?>
					<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $current, $value ); ?>><?php echo esc_html( $label ); ?></option>
				<?php endforeach; ?>
			</select>
			<?php submit_button( __( 'Filtrer', 'wc-efacturation-france' ), '', 'filter_action', false ); ?>
		</div>
		<?php
	}

	/**
	 * Message when no items.
	 */
	public function no_items(): void {
		esc_html_e( 'Aucune facture trouvee. Les factures seront generees automatiquement a chaque commande.', 'wc-efacturation-france' );
	}
}
