<?php
defined( 'ABSPATH' ) || exit;

/**
 * Invoice model — create, read, list invoices from DB.
 */
class WCEF_Invoice {

	/**
	 * Generate an invoice for an order.
	 *
	 * @param int $order_id WooCommerce order ID.
	 * @return string|WP_Error Invoice number on success, WP_Error on failure.
	 */
	public static function generate( int $order_id ) {
		global $wpdb;

		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return new WP_Error( 'invalid_order', __( 'Commande introuvable.', 'wc-efacturation-france' ) );
		}

		// Check if invoice already exists.
		$existing = self::get_by_order_id( $order_id );
		if ( $existing ) {
			return new WP_Error( 'already_exists', __( 'Une facture existe deja pour cette commande.', 'wc-efacturation-france' ) );
		}

		// Generate invoice number.
		$invoice_number = self::next_number();

		// Generate XML.
		$format      = get_option( 'wcef_invoice_format', 'facturx' );
		$xml_content = WCEF_XML_Generator::generate( $order, $invoice_number, $format );

		// Insert into DB.
		$table = $wpdb->prefix . 'wcef_invoices';
		$result = $wpdb->insert(
			$table,
			array(
				'order_id'       => $order_id,
				'invoice_number' => $invoice_number,
				'invoice_type'   => '380',
				'status'         => 'generated',
				'buyer_siren'    => $order->get_meta( '_billing_siren' ) ?: '',
				'buyer_tva'      => $order->get_meta( '_billing_tva_intra' ) ?: '',
				'total_ht'       => (float) $order->get_subtotal() + (float) $order->get_shipping_total(),
				'total_tax'      => (float) $order->get_total_tax(),
				'total_ttc'      => (float) $order->get_total(),
				'xml_content'    => $xml_content,
				'created_at'     => current_time( 'mysql' ),
			),
			array( '%d', '%s', '%s', '%s', '%s', '%s', '%f', '%f', '%f', '%s', '%s' )
		);

		if ( false === $result ) {
			return new WP_Error( 'db_error', __( 'Erreur lors de la creation de la facture.', 'wc-efacturation-france' ) );
		}

		// Save invoice ID to order meta.
		$invoice_id = $wpdb->insert_id;
		$order->update_meta_data( '_wcef_invoice_id', $invoice_id );
		$order->update_meta_data( '_wcef_invoice_number', $invoice_number );
		$order->save();

		// Generate and save the PDF to disk (immutable copy).
		$invoice_row = self::get( $invoice_id );
		if ( $invoice_row ) {
			$pdf_content = WCEF_PDF_Generator::generate( $invoice_row );
			if ( $pdf_content ) {
				$pdf_path = self::save_pdf( $invoice_number, $pdf_content );
				if ( $pdf_path ) {
					$wpdb->update(
						$table,
						array( 'pdf_path' => $pdf_path ),
						array( 'id' => $invoice_id ),
						array( '%s' ),
						array( '%d' )
					);
				}
			}
		}

		return $invoice_number;
	}

	/**
	 * Get invoice by ID.
	 */
	public static function get( int $id ): ?object {
		global $wpdb;
		$table = $wpdb->prefix . 'wcef_invoices';
		return $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table} WHERE id = %d", $id ) ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Get invoice by order ID.
	 */
	public static function get_by_order_id( int $order_id ): ?object {
		global $wpdb;
		$table = $wpdb->prefix . 'wcef_invoices';
		return $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table} WHERE order_id = %d", $order_id ) ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * List invoices with pagination.
	 *
	 * @param array $args Query args (status, per_page, offset, orderby, order, search).
	 * @return array [ 'items' => [], 'total' => int ]
	 */
	public static function list( array $args = array() ): array {
		global $wpdb;
		$table = $wpdb->prefix . 'wcef_invoices';

		$defaults = array(
			'status'   => '',
			'per_page' => 20,
			'offset'   => 0,
			'orderby'  => 'created_at',
			'order'    => 'DESC',
			'search'   => '',
		);
		$args = wp_parse_args( $args, $defaults );

		$where = array( '1=1' );
		$values = array();

		if ( $args['status'] ) {
			$where[]  = 'status = %s';
			$values[] = $args['status'];
		}

		if ( $args['search'] ) {
			$where[]  = '(invoice_number LIKE %s OR order_id = %d)';
			$values[] = '%' . $wpdb->esc_like( $args['search'] ) . '%';
			$values[] = absint( $args['search'] );
		}

		$where_sql = implode( ' AND ', $where );

		// Whitelist orderby.
		$allowed_orderby = array( 'created_at', 'invoice_number', 'total_ttc', 'status' );
		$orderby = in_array( $args['orderby'], $allowed_orderby, true ) ? $args['orderby'] : 'created_at';
		$order   = 'ASC' === strtoupper( $args['order'] ) ? 'ASC' : 'DESC';

		// Count.
		$count_sql = "SELECT COUNT(*) FROM {$table} WHERE {$where_sql}";
		if ( $values ) {
			$total = (int) $wpdb->get_var( $wpdb->prepare( $count_sql, ...$values ) ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		} else {
			$total = (int) $wpdb->get_var( $count_sql ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		}

		// Items.
		$query = "SELECT * FROM {$table} WHERE {$where_sql} ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d";
		$query_values = array_merge( $values, array( (int) $args['per_page'], (int) $args['offset'] ) );
		$items = $wpdb->get_results( $wpdb->prepare( $query, ...$query_values ) ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared

		return array(
			'items' => $items ?: array(),
			'total' => $total,
		);
	}

	/**
	 * Generate next sequential invoice number.
	 */
	private static function next_number(): string {
		$prefix = get_option( 'wcef_invoice_prefix', 'FA-' );
		$next   = (int) get_option( 'wcef_next_invoice_number', 1 );

		$number = $prefix . str_pad( (string) $next, 6, '0', STR_PAD_LEFT );

		update_option( 'wcef_next_invoice_number', $next + 1 );

		return $number;
	}

	/**
	 * Save PDF content to disk.
	 *
	 * @param string $invoice_number Invoice number (used for filename).
	 * @param string $pdf_content    Raw PDF binary.
	 * @return string|false Relative path from uploads dir, or false on failure.
	 */
	private static function save_pdf( string $invoice_number, string $pdf_content ) {
		$upload_dir = wp_upload_dir();
		$base_dir   = $upload_dir['basedir'] . '/wcef-invoices';

		// Create directory if it doesn't exist.
		if ( ! is_dir( $base_dir ) ) {
			wp_mkdir_p( $base_dir );
			// Protect directory from direct listing.
			file_put_contents( $base_dir . '/.htaccess', "Deny from all\n" ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
			file_put_contents( $base_dir . '/index.php', '<?php // Silence is golden.' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
		}

		$safe_name = sanitize_file_name( $invoice_number ) . '.pdf';
		$full_path = $base_dir . '/' . $safe_name;

		$written = file_put_contents( $full_path, $pdf_content ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
		if ( false === $written ) {
			return false;
		}

		// Return relative path from uploads basedir.
		return 'wcef-invoices/' . $safe_name;
	}

	/**
	 * Get the full filesystem path to a saved PDF.
	 *
	 * @param string $relative_path Relative path stored in DB.
	 * @return string Full filesystem path.
	 */
	public static function get_pdf_full_path( string $relative_path ): string {
		$upload_dir = wp_upload_dir();
		return $upload_dir['basedir'] . '/' . $relative_path;
	}

	/**
	 * Delete an invoice.
	 */
	public static function delete( int $id ): bool {
		global $wpdb;
		$table   = $wpdb->prefix . 'wcef_invoices';
		$invoice = self::get( $id );

		// Delete the PDF file from disk.
		if ( $invoice && ! empty( $invoice->pdf_path ) ) {
			$full_path = self::get_pdf_full_path( $invoice->pdf_path );
			if ( file_exists( $full_path ) ) {
				wp_delete_file( $full_path );
			}
		}

		return (bool) $wpdb->delete( $table, array( 'id' => $id ), array( '%d' ) );
	}

	/**
	 * Get all status labels.
	 */
	public static function get_status_labels(): array {
		return array(
			'generated'            => __( 'Generee', 'wc-efacturation-france' ),
			'pending_transmission' => __( 'En attente', 'wc-efacturation-france' ),
			'transmitted'          => __( 'Transmise', 'wc-efacturation-france' ),
			'accepted'             => __( 'Acceptee', 'wc-efacturation-france' ),
			'rejected'             => __( 'Rejetee', 'wc-efacturation-france' ),
			'delivered'            => __( 'Delivree', 'wc-efacturation-france' ),
			'error'                => __( 'Erreur', 'wc-efacturation-france' ),
		);
	}

	/**
	 * Get CSS color class suffix for each status.
	 */
	public static function get_status_colors(): array {
		return array(
			'generated'            => 'green',
			'pending_transmission' => 'amber',
			'transmitted'          => 'blue',
			'accepted'             => 'green',
			'rejected'             => 'red',
			'delivered'            => 'teal',
			'error'                => 'red',
		);
	}

	/**
	 * Update invoice status and log the change.
	 */
	public static function update_status( int $invoice_id, string $new_status, string $message = '' ): bool {
		global $wpdb;
		$table  = $wpdb->prefix . 'wcef_invoices';
		$result = $wpdb->update(
			$table,
			array(
				'status'            => $new_status,
				'status_updated_at' => current_time( 'mysql' ),
			),
			array( 'id' => $invoice_id ),
			array( '%s', '%s' ),
			array( '%d' )
		);

		self::log_status_change( $invoice_id, $new_status, $message );

		return (bool) $result;
	}

	/**
	 * Log a status change to the transmission log table.
	 */
	public static function log_status_change( int $invoice_id, string $status, string $message = '', string $chorus_response = '' ): void {
		global $wpdb;
		$table = $wpdb->prefix . 'wcef_transmission_log';
		$wpdb->insert(
			$table,
			array(
				'invoice_id'      => $invoice_id,
				'status'          => $status,
				'message'         => $message,
				'chorus_response' => $chorus_response,
				'created_at'      => current_time( 'mysql' ),
			),
			array( '%d', '%s', '%s', '%s', '%s' )
		);
	}

	/**
	 * Get the status timeline for an invoice (ordered chronologically).
	 */
	public static function get_status_timeline( int $invoice_id ): array {
		global $wpdb;
		$table = $wpdb->prefix . 'wcef_transmission_log';
		$results = $wpdb->get_results( $wpdb->prepare(
			"SELECT * FROM {$table} WHERE invoice_id = %d ORDER BY created_at ASC", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$invoice_id
		) );
		return $results ?: array();
	}
}
