<?php
defined( 'ABSPATH' ) || exit;

/**
 * License management — verifies API key, caches plan+routing, gates features.
 */
class WCEF_License {

	const API_URL       = 'https://efacturx.com/api/license-verify.php';
	const TRANSIENT_KEY = 'wcef_license_data';
	const CACHE_TTL     = 12 * HOUR_IN_SECONDS;

	/**
	 * Base plan hierarchy (lower index = fewer features).
	 */
	private static array $plan_hierarchy = array(
		'starter',
		'core',
		'partner',
	);

	/**
	 * Routing tier hierarchy.
	 */
	private static array $routing_hierarchy = array(
		'none',
		'routing_s',
		'routing_m',
		'routing_l',
		'routing_xl',
	);

	/**
	 * Feature → minimum base plan required.
	 */
	private static array $feature_plans = array(
		'generation'       => 'starter',
		'pdf_xml_export'   => 'starter',
		'b2b_checkout'     => 'starter',
		'dashboard'        => 'starter',
		'auto_generation'  => 'core',
		'export_comptable' => 'core',
		'api_webhooks'     => 'core',
		'multi_boutiques'  => 'partner',
	);

	/**
	 * Feature → minimum routing tier required.
	 */
	private static array $feature_routing = array(
		'routage_pa'             => 'routing_s',
		'statuts_temps_reel'     => 'routing_s',
		'reception_fournisseurs' => 'routing_s',
		'routing_history'        => 'routing_s',
	);

	/**
	 * Routing limits per tier.
	 */
	private static array $routing_limits = array(
		'none'       => array( 'routed_per_month' => 0, 'overage_rate' => 0 ),
		'routing_s'  => array( 'routed_per_month' => 200, 'overage_rate' => 0.15 ),
		'routing_m'  => array( 'routed_per_month' => 600, 'overage_rate' => 0.12 ),
		'routing_l'  => array( 'routed_per_month' => 2000, 'overage_rate' => 0.08 ),
		'routing_xl' => array( 'routed_per_month' => -1, 'overage_rate' => 0 ),
	);

	/**
	 * Backward compat: old plan names → new plan + routing.
	 */
	private static array $plan_migration = array(
		'pro_byo'     => array( 'plan' => 'core', 'routing' => 'none' ),
		'pro_routage' => array( 'plan' => 'core', 'routing' => 'routing_s' ),
		'agency'      => array( 'plan' => 'partner', 'routing' => 'none' ),
	);

	/**
	 * Human-readable plan labels.
	 */
	public static function get_plan_labels(): array {
		return array(
			'starter' => 'Gratuit',
			'core'    => 'Gratuit',
			'partner' => 'Agence',
		);
	}

	/**
	 * Human-readable routing labels.
	 */
	public static function get_routing_labels(): array {
		return array(
			'none'       => '',
			'routing_s'  => 'Essentiel',
			'routing_m'  => 'Business',
			'routing_l'  => 'Premium',
			'routing_xl' => 'Agence',
		);
	}

	/**
	 * Get the full human-readable label (e.g. "Essentiel", "Gratuit").
	 */
	public static function get_full_plan_label(): string {
		$data    = self::verify();
		$plan    = $data['plan'];
		$routing = $data['routing'] ?? 'none';
		$labels  = self::get_plan_labels();
		$rlabels = self::get_routing_labels();
		if ( 'none' !== $routing && ! empty( $rlabels[ $routing ] ) ) {
			return $rlabels[ $routing ];
		}
		return $labels[ $plan ] ?? 'Gratuit';
	}

	/**
	 * Verify the license via API or cached transient.
	 *
	 * @return array{ valid: bool, plan: string, routing: string, limits: array, expires: string, error?: string }
	 */
	public static function verify( bool $force = false ): array {
		$key = get_option( 'wcef_license_key', '' );

		// No key = free plan.
		if ( empty( $key ) ) {
			return self::starter_response();
		}

		// Check cache first.
		if ( ! $force ) {
			$cached = get_transient( self::TRANSIENT_KEY );
			if ( false !== $cached && is_array( $cached ) ) {
				return $cached;
			}
		}

		// Call the API.
		$response = wp_remote_post( self::API_URL, array(
			'timeout' => 15,
			'body'    => array(
				'license_key' => $key,
				'site_url'    => home_url(),
			),
		) );

		if ( is_wp_error( $response ) ) {
			$cached = get_transient( self::TRANSIENT_KEY );
			if ( false !== $cached && is_array( $cached ) ) {
				return $cached;
			}
			return array_merge( self::starter_response(), array( 'error' => $response->get_error_message() ) );
		}

		$code = wp_remote_retrieve_response_code( $response );
		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( 200 !== $code || empty( $body ) || ! is_array( $body ) ) {
			return array_merge( self::starter_response(), array( 'error' => 'Reponse invalide du serveur de licences.' ) );
		}

		// Normalize.
		$plan    = sanitize_text_field( $body['plan'] ?? 'starter' );
		$routing = sanitize_text_field( $body['routing'] ?? 'none' );

		// Backward compat migration.
		if ( isset( self::$plan_migration[ $plan ] ) ) {
			$migration = self::$plan_migration[ $plan ];
			$plan      = $migration['plan'];
			$routing   = $migration['routing'];
		}

		$data = array(
			'valid'   => ! empty( $body['valid'] ),
			'plan'    => $plan,
			'routing' => $routing,
			'limits'  => $body['limits'] ?? self::$routing_limits['none'],
			'expires' => sanitize_text_field( $body['expires'] ?? '' ),
		);

		if ( ! $data['valid'] ) {
			$data['plan']    = 'starter';
			$data['routing'] = 'none';
			$data['limits']  = self::$routing_limits['none'];
			$data['error']   = sanitize_text_field( $body['error'] ?? 'Licence invalide.' );
		}

		// Cache the result.
		set_transient( self::TRANSIENT_KEY, $data, self::CACHE_TTL );

		return $data;
	}

	/**
	 * Get the current active base plan.
	 */
	public static function get_plan(): string {
		$data = self::verify();
		return $data['plan'];
	}

	/**
	 * Get the current routing tier.
	 */
	public static function get_routing(): string {
		$data = self::verify();
		return $data['routing'] ?? 'none';
	}

	/**
	 * Check if a feature is available for the current plan + routing.
	 */
	public static function can( string $feature ): bool {
		$data    = self::verify();
		$plan    = $data['plan'];
		$routing = $data['routing'] ?? 'none';

		// Check base plan features.
		if ( isset( self::$feature_plans[ $feature ] ) ) {
			$required_plan  = self::$feature_plans[ $feature ];
			$plan_index     = array_search( $plan, self::$plan_hierarchy, true );
			$required_index = array_search( $required_plan, self::$plan_hierarchy, true );
			if ( false === $plan_index || false === $required_index ) {
				return false;
			}
			return $plan_index >= $required_index;
		}

		// Check routing features.
		if ( isset( self::$feature_routing[ $feature ] ) ) {
			$required_routing = self::$feature_routing[ $feature ];
			$routing_index    = array_search( $routing, self::$routing_hierarchy, true );
			$required_index   = array_search( $required_routing, self::$routing_hierarchy, true );
			if ( false === $routing_index || false === $required_index ) {
				return false;
			}
			return $routing_index >= $required_index;
		}

		return false;
	}

	/**
	 * Get the limits for the current routing tier.
	 */
	public static function get_limits(): array {
		$data    = self::verify();
		$routing = $data['routing'] ?? 'none';
		return self::$routing_limits[ $routing ] ?? self::$routing_limits['none'];
	}

	/**
	 * Force a re-check of the license (clears cache).
	 */
	public static function refresh(): array {
		delete_transient( self::TRANSIENT_KEY );
		return self::verify( true );
	}

	/**
	 * Deactivate the license (clear key and cache).
	 */
	public static function deactivate(): void {
		delete_option( 'wcef_license_key' );
		delete_transient( self::TRANSIENT_KEY );
	}

	/**
	 * Default response for free plan.
	 */
	private static function starter_response(): array {
		return array(
			'valid'   => true,
			'plan'    => 'starter',
			'routing' => 'none',
			'limits'  => self::$routing_limits['none'],
			'expires' => '',
		);
	}
}
