<?php
defined( 'ABSPATH' ) || exit;

require_once WCEF_PLUGIN_DIR . 'lib/fpdf/fpdf.php';

/**
 * Generates elegant PDF invoices using FPDF.
 * Clean layout with proper logo sizing, balanced header, and legal footer.
 */
class WCEF_PDF_Generator {

	/** @var float Left margin. */
	private const M_LEFT = 15;

	/** @var float Right margin. */
	private const M_RIGHT = 15;

	/** @var float Usable width (A4 = 210mm). */
	private const W = 180; // 210 - 15 - 15

	/**
	 * Generate a PDF binary string.
	 *
	 * @param object $invoice Invoice row from DB.
	 * @return string PDF binary content.
	 */
	public static function generate( object $invoice ): string {
		$order = wc_get_order( $invoice->order_id );
		if ( ! $order ) {
			return '';
		}

		$data = self::gather_data( $invoice, $order );

		$pdf = new FPDF( 'P', 'mm', 'A4' );
		$pdf->SetAutoPageBreak( true, 30 );
		$pdf->SetMargins( self::M_LEFT, 15, self::M_RIGHT );
		$pdf->AddPage();

		self::draw_header( $pdf, $data );
		self::draw_parties( $pdf, $data );
		self::draw_line_items( $pdf, $data );
		self::draw_totals( $pdf, $data );
		self::draw_payment_terms( $pdf, $data );
		self::draw_footer( $pdf, $data );

		return $pdf->Output( 'S' );
	}

	/**
	 * Gather all data needed for the invoice.
	 */
	private static function gather_data( object $invoice, $order ): array {
		$accent    = self::hex_to_rgb( get_option( 'wcef_design_accent', '#6366f1' ) );
		$font      = self::map_font( get_option( 'wcef_design_font', 'Helvetica Neue' ) );
		$logo_id   = absint( get_option( 'wcef_logo_id', 0 ) );
		$logo_path = $logo_id ? get_attached_file( $logo_id ) : '';
		$logo_max  = (float) get_option( 'wcef_design_logo_max_height', '18' );

		// Clamp logo max height.
		$logo_max = max( 8, min( 40, $logo_max ) );

		$seller_name    = get_option( 'wcef_company_name', get_bloginfo( 'name' ) );
		$seller_address = get_option( 'wcef_address', '' );
		$seller_pc      = get_option( 'wcef_postcode', '' );
		$seller_city    = get_option( 'wcef_city', '' );
		$seller_siren   = get_option( 'wcef_siren', '' );
		$seller_siret   = get_option( 'wcef_siret', '' );
		$seller_tva     = get_option( 'wcef_tva_intra', '' );
		$seller_legal   = get_option( 'wcef_legal_form', '' );
		$rcs            = get_option( 'wcef_rcs', '' );
		$capital        = get_option( 'wcef_capital', '' );
		$vat_regime     = get_option( 'wcef_vat_regime', 'standard' );
		$option_debits  = get_option( 'wcef_option_debits', 'no' );
		$payment_delay  = get_option( 'wcef_payment_delay', '30' );
		$penalty_rate   = get_option( 'wcef_penalty_rate', '3x le taux d\'interet legal' );
		$early_payment  = get_option( 'wcef_early_payment', 'Pas d\'escompte pour paiement anticipe' );
		$legal_mentions = get_option( 'wcef_legal_mentions', '' );
		$show_logo      = get_option( 'wcef_design_show_logo', 'yes' );
		$show_footer    = get_option( 'wcef_design_show_footer', 'yes' );
		$bank_iban      = get_option( 'wcef_bank_iban', '' );
		$bank_bic       = get_option( 'wcef_bank_bic', '' );
		$bank_name      = get_option( 'wcef_bank_name', '' );
		$footer_extra   = get_option( 'wcef_footer_extra', '' );

		$buyer_name = $order->get_billing_company()
			?: trim( $order->get_billing_first_name() . ' ' . $order->get_billing_last_name() );

		$inv_date  = date_i18n( 'd/m/Y', strtotime( $invoice->created_at ) );
		$sale_date = $order->get_date_created() ? $order->get_date_created()->date_i18n( 'd/m/Y' ) : $inv_date;

		// Build auto-footer from company info.
		$auto_footer = self::build_auto_footer(
			$seller_name, $seller_legal, $seller_address, $seller_pc, $seller_city,
			$seller_siren, $seller_siret, $seller_tva, $rcs, $capital, $vat_regime
		);

		return compact(
			'order', 'invoice', 'accent', 'font', 'logo_path', 'logo_max',
			'seller_name', 'seller_address', 'seller_pc', 'seller_city',
			'seller_siren', 'seller_siret', 'seller_tva', 'seller_legal',
			'rcs', 'capital', 'vat_regime', 'option_debits',
			'payment_delay', 'penalty_rate', 'early_payment', 'legal_mentions',
			'show_logo', 'show_footer', 'bank_iban', 'bank_bic', 'bank_name', 'footer_extra',
			'buyer_name', 'inv_date', 'sale_date', 'auto_footer'
		);
	}

	/**
	 * Build auto-generated footer from company settings.
	 */
	private static function build_auto_footer(
		string $name, string $legal_form, string $address, string $postcode, string $city,
		string $siren, string $siret, string $tva, string $rcs, string $capital, string $vat_regime
	): string {
		$parts = array();

		// Company name + legal form.
		$company_line = $name;
		if ( $legal_form ) {
			$company_line .= ' - ' . $legal_form;
		}
		if ( $capital ) {
			$company_line .= ' au capital de ' . $capital;
		}
		$parts[] = $company_line;

		// Address.
		$addr = trim( $address );
		if ( $postcode || $city ) {
			$addr .= ( $addr ? ' - ' : '' ) . trim( $postcode . ' ' . $city );
		}
		if ( $addr ) {
			$parts[] = $addr;
		}

		// Legal identifiers.
		$ids = array();
		if ( $rcs ) {
			$ids[] = $rcs;
		}
		if ( $siren ) {
			$ids[] = 'SIREN ' . $siren;
		}
		if ( $siret ) {
			$ids[] = 'SIRET ' . $siret;
		}
		if ( $tva && 'franchise' !== $vat_regime ) {
			$ids[] = 'TVA ' . $tva;
		}
		if ( $ids ) {
			$parts[] = implode( ' - ', $ids );
		}

		return implode( "\n", $parts );
	}

	// ──────────────────────────────────────────────
	//  HEADER
	// ──────────────────────────────────────────────

	private static function draw_header( FPDF $pdf, array $d ): void {
		$w      = self::W;
		$font   = $d['font'];
		$accent = $d['accent'];

		$start_y = $pdf->GetY();

		// ── Right side: FACTURE title + number + dates ──
		// Draw this first to know its height, then draw logo+company on the left.
		$right_x = self::M_LEFT + $w * 0.55;
		$right_w = $w * 0.45;

		$pdf->SetY( $start_y );
		$pdf->SetFont( $font, 'B', 22 );
		$pdf->SetTextColor( $accent[0], $accent[1], $accent[2] );
		$pdf->Cell( $w, 9, 'FACTURE', 0, 1, 'R' );

		$pdf->SetFont( $font, '', 10 );
		$pdf->SetTextColor( 55, 65, 81 );
		$pdf->Cell( $w, 5, self::utf8( $d['invoice']->invoice_number ), 0, 1, 'R' );

		$pdf->Ln( 2 );
		$pdf->SetFont( $font, '', 8 );
		$pdf->SetTextColor( 107, 114, 128 );
		$pdf->Cell( $w, 4, self::utf8( 'Date de facture : ' . $d['inv_date'] ), 0, 1, 'R' );
		if ( $d['sale_date'] !== $d['inv_date'] ) {
			$pdf->Cell( $w, 4, self::utf8( 'Date de vente : ' . $d['sale_date'] ), 0, 1, 'R' );
		}

		$right_bottom = $pdf->GetY();

		// ── Left side: Logo + company info ──
		$pdf->SetY( $start_y );
		$left_w = $w * 0.50;

		$logo_rendered = false;
		if ( 'yes' === $d['show_logo'] && $d['logo_path'] && file_exists( $d['logo_path'] ) ) {
			$ext = strtolower( pathinfo( $d['logo_path'], PATHINFO_EXTENSION ) );
			if ( in_array( $ext, array( 'jpg', 'jpeg', 'png' ), true ) ) {
				// Get image dimensions to compute width from max height.
				$img_info = getimagesize( $d['logo_path'] );
				if ( $img_info ) {
					$img_w = $img_info[0];
					$img_h = $img_info[1];
					$max_h = $d['logo_max'];
					$ratio = $img_w / max( 1, $img_h );
					$draw_h = $max_h;
					$draw_w = $draw_h * $ratio;

					// Cap width at 50mm.
					if ( $draw_w > 50 ) {
						$draw_w = 50;
						$draw_h = $draw_w / $ratio;
					}

					$pdf->Image( $d['logo_path'], self::M_LEFT, $start_y, $draw_w, $draw_h );
					$logo_rendered = true;
					$pdf->SetY( $start_y + $draw_h + 3 );
				}
			}
		}

		// Company name.
		$pdf->SetFont( $font, 'B', 11 );
		$pdf->SetTextColor( 17, 24, 39 );
		$pdf->Cell( $left_w, 5, self::utf8( $d['seller_name'] ), 0, 1 );

		// Address.
		$pdf->SetFont( $font, '', 8 );
		$pdf->SetTextColor( 75, 85, 99 );
		if ( $d['seller_address'] ) {
			$pdf->Cell( $left_w, 3.5, self::utf8( $d['seller_address'] ), 0, 1 );
		}
		if ( $d['seller_pc'] || $d['seller_city'] ) {
			$pdf->Cell( $left_w, 3.5, self::utf8( trim( $d['seller_pc'] . ' ' . $d['seller_city'] ) ), 0, 1 );
		}

		// Legal line: legal form, capital, RCS.
		$pdf->SetFont( $font, '', 7 );
		$pdf->SetTextColor( 130, 130, 140 );
		$id_line_1 = array();
		if ( $d['seller_legal'] ) {
			$id_line_1[] = $d['seller_legal'];
		}
		if ( $d['capital'] ) {
			$id_line_1[] = 'Capital : ' . $d['capital'];
		}
		if ( $d['rcs'] ) {
			$id_line_1[] = $d['rcs'];
		}
		if ( $id_line_1 ) {
			$pdf->Cell( $left_w, 3.5, self::utf8( implode( ' - ', $id_line_1 ) ), 0, 1 );
		}

		// SIREN / SIRET / TVA.
		$id_line_2 = array();
		if ( $d['seller_siren'] ) {
			$id_line_2[] = 'SIREN : ' . $d['seller_siren'];
		}
		if ( $d['seller_siret'] ) {
			$id_line_2[] = 'SIRET : ' . $d['seller_siret'];
		}
		if ( $d['seller_tva'] && 'franchise' !== $d['vat_regime'] ) {
			$id_line_2[] = 'TVA : ' . $d['seller_tva'];
		}
		if ( $id_line_2 ) {
			$pdf->Cell( $left_w, 3.5, self::utf8( implode( ' | ', $id_line_2 ) ), 0, 1 );
		}

		$left_bottom = $pdf->GetY();

		// Separator line.
		$line_y = max( $left_bottom, $right_bottom ) + 5;
		$pdf->SetY( $line_y );
		$pdf->SetDrawColor( 229, 231, 235 );
		$pdf->SetLineWidth( 0.3 );
		$pdf->Line( self::M_LEFT, $line_y, self::M_LEFT + $w, $line_y );
		$pdf->SetY( $line_y + 6 );
	}

	// ──────────────────────────────────────────────
	//  BUYER + ORDER INFO
	// ──────────────────────────────────────────────

	private static function draw_parties( FPDF $pdf, array $d ): void {
		$w      = self::W;
		$font   = $d['font'];
		$accent = $d['accent'];
		$order  = $d['order'];

		$start_y = $pdf->GetY();
		$col_w   = $w / 2;

		// ── Left: Buyer ──
		$pdf->SetFont( $font, 'B', 7 );
		$pdf->SetTextColor( $accent[0], $accent[1], $accent[2] );
		$pdf->Cell( $col_w, 4, 'FACTURER A', 0, 1 );
		$pdf->Ln( 1 );

		$pdf->SetFont( $font, 'B', 9 );
		$pdf->SetTextColor( 17, 24, 39 );
		$pdf->Cell( $col_w, 5, self::utf8( $d['buyer_name'] ), 0, 1 );

		$pdf->SetFont( $font, '', 8 );
		$pdf->SetTextColor( 75, 85, 99 );
		if ( $order->get_billing_address_1() ) {
			$pdf->Cell( $col_w, 3.5, self::utf8( $order->get_billing_address_1() ), 0, 1 );
		}
		if ( $order->get_billing_postcode() || $order->get_billing_city() ) {
			$pdf->Cell( $col_w, 3.5, self::utf8( trim( $order->get_billing_postcode() . ' ' . $order->get_billing_city() ) ), 0, 1 );
		}
		$country_name = WC()->countries->countries[ $order->get_billing_country() ] ?? $order->get_billing_country();
		if ( $country_name ) {
			$pdf->Cell( $col_w, 3.5, self::utf8( $country_name ), 0, 1 );
		}

		// B2B identifiers.
		$b_siren = $order->get_meta( '_billing_siren' );
		$b_siret = $order->get_meta( '_billing_siret' );
		$b_tva   = $order->get_meta( '_billing_tva_intra' );
		if ( $b_siren || $b_siret || $b_tva ) {
			$pdf->Ln( 1 );
			$pdf->SetFont( $font, '', 7 );
			$pdf->SetTextColor( 130, 130, 140 );
			$b_parts = array();
			if ( $b_siren ) {
				$b_parts[] = 'SIREN : ' . $b_siren;
			}
			if ( $b_siret ) {
				$b_parts[] = 'SIRET : ' . $b_siret;
			}
			if ( $b_tva ) {
				$b_parts[] = 'TVA : ' . $b_tva;
			}
			$pdf->Cell( $col_w, 3.5, self::utf8( implode( ' | ', $b_parts ) ), 0, 1 );
		}

		$buyer_bottom = $pdf->GetY();

		// ── Right: Order info ──
		$pdf->SetY( $start_y );

		$pdf->SetFont( $font, 'B', 7 );
		$pdf->SetTextColor( $accent[0], $accent[1], $accent[2] );
		$pdf->Cell( $w, 4, 'COMMANDE', 0, 1, 'R' );
		$pdf->Ln( 1 );

		$pdf->SetFont( $font, '', 8 );
		$pdf->SetTextColor( 75, 85, 99 );

		$po_number   = $order->get_meta( '_po_number' );
		$order_lines = array();
		$order_lines[] = 'N' . "\xC2\xB0" . ' ' . $order->get_order_number();
		if ( $po_number ) {
			$order_lines[] = 'Ref : ' . $po_number;
		}
		$order_lines[] = 'Date : ' . ( $order->get_date_created() ? $order->get_date_created()->date_i18n( 'd/m/Y' ) : '' );
		$order_lines[] = 'Paiement : ' . $order->get_payment_method_title();

		foreach ( $order_lines as $ol ) {
			$pdf->Cell( $w, 4, self::utf8( $ol ), 0, 1, 'R' );
		}

		$pdf->SetY( max( $buyer_bottom, $pdf->GetY() ) + 8 );
	}

	// ──────────────────────────────────────────────
	//  LINE ITEMS TABLE
	// ──────────────────────────────────────────────

	private static function draw_line_items( FPDF $pdf, array $d ): void {
		$w      = self::W;
		$font   = $d['font'];
		$accent = $d['accent'];
		$order  = $d['order'];

		// Column widths: # | Description | Ref | Qty | Unit Price | VAT | Total.
		$cols   = array( 10, 64, 20, 16, 26, 18, 26 );
		$aligns = array( 'C', 'L', 'L', 'R', 'R', 'R', 'R' );

		// ── Header row ──
		$pdf->SetFont( $font, 'B', 7 );
		$pdf->SetFillColor( $accent[0], $accent[1], $accent[2] );
		$pdf->SetTextColor( 255, 255, 255 );
		$pdf->SetDrawColor( $accent[0], $accent[1], $accent[2] );

		$headers = array( '#', 'DESCRIPTION', 'REF', 'QTE', 'P.U. HT', 'TVA', 'TOTAL HT' );
		$header_h = 7;
		for ( $i = 0; $i < count( $headers ); $i++ ) {
			$pdf->Cell( $cols[ $i ], $header_h, $headers[ $i ], 0, 0, $aligns[ $i ], true );
		}
		$pdf->Ln();

		// ── Data rows ──
		$pdf->SetFont( $font, '', 7.5 );
		$pdf->SetDrawColor( 240, 240, 240 );
		$pdf->SetLineWidth( 0.1 );

		$line_num   = 0;
		$row_height = 6;

		foreach ( $order->get_items() as $item ) {
			$line_num++;
			$product    = $item->get_product();
			$unit_price = (float) $item->get_subtotal() / max( 1, $item->get_quantity() );
			$tax_rate   = 0;
			if ( (float) $item->get_subtotal() > 0 ) {
				$tax_rate = round( ( (float) $item->get_subtotal_tax() / (float) $item->get_subtotal() ) * 100, 1 );
			}

			// Page break check.
			if ( $pdf->GetY() + $row_height > $pdf->GetPageHeight() - 50 ) {
				$pdf->AddPage();
			}

			// Alternate row shading.
			$fill = ( $line_num % 2 === 0 );
			if ( $fill ) {
				$pdf->SetFillColor( 249, 250, 251 );
			}

			$pdf->SetTextColor( 33, 33, 33 );
			$row = array(
				(string) $line_num,
				self::utf8( mb_strimwidth( $item->get_name(), 0, 44, '...' ) ),
				$product ? $product->get_sku() : '-',
				(string) $item->get_quantity(),
				self::fmt( $unit_price ),
				$tax_rate . '%',
				self::fmt( (float) $item->get_subtotal() ),
			);

			for ( $i = 0; $i < count( $row ); $i++ ) {
				$pdf->Cell( $cols[ $i ], $row_height, $row[ $i ], 'B', 0, $aligns[ $i ], $fill );
			}
			$pdf->Ln();
		}

		// ── Shipping row ──
		if ( (float) $order->get_shipping_total() > 0 ) {
			$line_num++;
			$ship_tax_rate = 0;
			if ( (float) $order->get_shipping_total() > 0 ) {
				$ship_tax_rate = round( ( (float) $order->get_shipping_tax() / (float) $order->get_shipping_total() ) * 100, 1 );
			}

			$fill = ( $line_num % 2 === 0 );
			if ( $fill ) {
				$pdf->SetFillColor( 249, 250, 251 );
			}

			$pdf->SetTextColor( 33, 33, 33 );
			$ship_row = array(
				(string) $line_num,
				'Frais de livraison',
				'-',
				'1',
				self::fmt( (float) $order->get_shipping_total() ),
				$ship_tax_rate . '%',
				self::fmt( (float) $order->get_shipping_total() ),
			);

			for ( $i = 0; $i < count( $ship_row ); $i++ ) {
				$pdf->Cell( $cols[ $i ], $row_height, $ship_row[ $i ], 'B', 0, $aligns[ $i ], $fill );
			}
			$pdf->Ln();
		}

		$pdf->Ln( 4 );
	}

	// ──────────────────────────────────────────────
	//  TOTALS
	// ──────────────────────────────────────────────

	private static function draw_totals( FPDF $pdf, array $d ): void {
		$w      = self::W;
		$font   = $d['font'];
		$accent = $d['accent'];
		$order  = $d['order'];

		$totals_w  = 75;
		$totals_x  = self::M_LEFT + $w - $totals_w;
		$total_ht  = (float) $order->get_subtotal() + (float) $order->get_shipping_total();
		$total_tax = (float) $order->get_total_tax();
		$total_ttc = (float) $order->get_total();

		// Check page break for totals + legal block (~65mm).
		if ( $pdf->GetY() + 65 > $pdf->GetPageHeight() - 30 ) {
			$pdf->AddPage();
		}

		$row_h = 6;

		// Total HT.
		$pdf->SetFont( $font, '', 8 );
		$pdf->SetTextColor( 107, 114, 128 );
		$pdf->SetX( $totals_x );
		$pdf->Cell( 42, $row_h, 'Total HT', 0, 0, 'L' );
		$pdf->SetTextColor( 33, 33, 33 );
		$pdf->Cell( 33, $row_h, self::fmt( $total_ht ), 0, 1, 'R' );

		// Discount.
		if ( (float) $order->get_discount_total() > 0 ) {
			$pdf->SetTextColor( 107, 114, 128 );
			$pdf->SetX( $totals_x );
			$pdf->Cell( 42, $row_h, 'Remise', 0, 0, 'L' );
			$pdf->SetTextColor( 33, 33, 33 );
			$pdf->Cell( 33, $row_h, '-' . self::fmt( (float) $order->get_discount_total() ), 0, 1, 'R' );
		}

		// TVA.
		$pdf->SetTextColor( 107, 114, 128 );
		$pdf->SetX( $totals_x );
		$pdf->Cell( 42, $row_h, 'TVA', 0, 0, 'L' );
		$pdf->SetTextColor( 33, 33, 33 );
		$pdf->Cell( 33, $row_h, self::fmt( $total_tax ), 0, 1, 'R' );

		// Separator before grand total.
		$pdf->Ln( 1 );
		$pdf->SetDrawColor( $accent[0], $accent[1], $accent[2] );
		$pdf->SetLineWidth( 0.4 );
		$pdf->Line( $totals_x, $pdf->GetY(), $totals_x + $totals_w, $pdf->GetY() );
		$pdf->Ln( 2 );

		// Grand total with accent background.
		$pdf->SetFont( $font, 'B', 10 );
		$pdf->SetFillColor(
			min( 255, $accent[0] + (int) ( ( 255 - $accent[0] ) * 0.92 ) ),
			min( 255, $accent[1] + (int) ( ( 255 - $accent[1] ) * 0.92 ) ),
			min( 255, $accent[2] + (int) ( ( 255 - $accent[2] ) * 0.92 ) )
		);
		$pdf->SetTextColor( $accent[0], $accent[1], $accent[2] );
		$pdf->SetX( $totals_x );
		$pdf->Cell( 42, 8, 'Total TTC', 0, 0, 'L', true );
		$pdf->Cell( 33, 8, self::fmt( $total_ttc ), 0, 1, 'R', true );

		$pdf->Ln( 4 );

		// ── VAT regime mention ──
		$vat_mention = self::get_vat_mention( $d['vat_regime'] );
		if ( $vat_mention ) {
			$pdf->SetFont( $font, 'B', 7 );
			$pdf->SetTextColor( $accent[0], $accent[1], $accent[2] );
			$pdf->Cell( $w, 4, self::utf8( $vat_mention ), 0, 1, 'L' );
		}

		// Option debits.
		if ( 'yes' === $d['option_debits'] ) {
			$pdf->SetFont( $font, 'I', 7 );
			$pdf->SetTextColor( 130, 130, 140 );
			$pdf->Cell( $w, 3.5, self::utf8( 'Option pour le paiement de la TVA d\'apres les debits' ), 0, 1, 'L' );
		}

		$pdf->Ln( 2 );
	}

	// ──────────────────────────────────────────────
	//  PAYMENT TERMS
	// ──────────────────────────────────────────────

	private static function draw_payment_terms( FPDF $pdf, array $d ): void {
		$w    = self::W;
		$font = $d['font'];

		$pdf->SetDrawColor( 229, 231, 235 );
		$pdf->SetLineWidth( 0.15 );
		$pdf->Line( self::M_LEFT, $pdf->GetY(), self::M_LEFT + $w, $pdf->GetY() );
		$pdf->Ln( 3 );

		$pdf->SetFont( $font, 'B', 7 );
		$pdf->SetTextColor( 55, 65, 81 );
		$pdf->Cell( $w, 3.5, 'CONDITIONS DE PAIEMENT', 0, 1 );
		$pdf->Ln( 1 );

		$pdf->SetFont( $font, '', 7 );
		$pdf->SetTextColor( 75, 85, 99 );

		$delay_label = self::get_delay_label( $d['payment_delay'] );
		$pdf->Cell( $w, 3.5, self::utf8( 'Echeance : ' . $delay_label ), 0, 1 );

		if ( $d['early_payment'] ) {
			$pdf->Cell( $w, 3.5, self::utf8( 'Escompte : ' . $d['early_payment'] ), 0, 1 );
		}
		if ( $d['penalty_rate'] ) {
			$pdf->Cell( $w, 3.5, self::utf8( 'Penalites de retard : ' . $d['penalty_rate'] ), 0, 1 );
		}

		$pdf->Cell( $w, 3.5, self::utf8( 'Indemnite forfaitaire pour frais de recouvrement : 40,00 EUR' ), 0, 1 );

		$pdf->Ln( 2 );

		// ── Custom legal mentions ──
		if ( $d['legal_mentions'] ) {
			$pdf->SetFont( $font, '', 6.5 );
			$pdf->SetTextColor( 156, 163, 175 );
			$pdf->MultiCell( $w, 3, self::utf8( $d['legal_mentions'] ), 0, 'L' );
			$pdf->Ln( 1 );
		}
	}

	// ──────────────────────────────────────────────
	//  FOOTER (auto-generated legal line)
	// ──────────────────────────────────────────────

	private static function draw_footer( FPDF $pdf, array $d ): void {
		$w      = self::W;
		$font   = $d['font'];
		$accent = $d['accent'];

		$has_bank = $d['bank_iban'] || $d['bank_bic'] || $d['bank_name'];

		// Structured banking block.
		if ( 'yes' === $d['show_footer'] && $has_bank ) {
			$pdf->SetDrawColor( 229, 231, 235 );
			$pdf->SetLineWidth( 0.15 );
			$pdf->Line( self::M_LEFT, $pdf->GetY(), self::M_LEFT + $w, $pdf->GetY() );
			$pdf->Ln( 3 );

			$pdf->SetFont( $font, 'B', 7 );
			$pdf->SetTextColor( $accent[0], $accent[1], $accent[2] );
			$pdf->Cell( $w, 3.5, 'COORDONNEES BANCAIRES', 0, 1, 'L' );
			$pdf->Ln( 1 );

			$pdf->SetFont( $font, '', 7 );
			$pdf->SetTextColor( 75, 85, 99 );

			$label_w = 18;
			$value_w = $w - $label_w;

			if ( $d['bank_iban'] ) {
				$pdf->SetFont( $font, 'B', 7 );
				$pdf->Cell( $label_w, 3.5, 'IBAN', 0, 0, 'L' );
				$pdf->SetFont( $font, '', 7 );
				$pdf->Cell( $value_w, 3.5, self::utf8( $d['bank_iban'] ), 0, 1, 'L' );
			}
			if ( $d['bank_bic'] ) {
				$pdf->SetFont( $font, 'B', 7 );
				$pdf->Cell( $label_w, 3.5, 'BIC', 0, 0, 'L' );
				$pdf->SetFont( $font, '', 7 );
				$pdf->Cell( $value_w, 3.5, self::utf8( $d['bank_bic'] ), 0, 1, 'L' );
			}
			if ( $d['bank_name'] ) {
				$pdf->SetFont( $font, 'B', 7 );
				$pdf->Cell( $label_w, 3.5, 'Banque', 0, 0, 'L' );
				$pdf->SetFont( $font, '', 7 );
				$pdf->Cell( $value_w, 3.5, self::utf8( $d['bank_name'] ), 0, 1, 'L' );
			}

			$pdf->Ln( 1 );
		}

		// Extra free-text footer.
		if ( 'yes' === $d['show_footer'] && $d['footer_extra'] ) {
			if ( ! $has_bank ) {
				$pdf->SetDrawColor( 229, 231, 235 );
				$pdf->SetLineWidth( 0.15 );
				$pdf->Line( self::M_LEFT, $pdf->GetY(), self::M_LEFT + $w, $pdf->GetY() );
				$pdf->Ln( 2.5 );
			}

			$pdf->SetFont( $font, '', 7 );
			$pdf->SetTextColor( 130, 130, 140 );
			$pdf->MultiCell( $w, 3.5, self::utf8( $d['footer_extra'] ), 0, 'C' );
			$pdf->Ln( 1 );
		}

		// Auto-generated legal footer (always shown).
		if ( $d['auto_footer'] ) {
			$pdf->SetDrawColor( 200, 200, 200 );
			$pdf->SetLineWidth( 0.1 );
			$pdf->Line( self::M_LEFT, $pdf->GetY(), self::M_LEFT + $w, $pdf->GetY() );
			$pdf->Ln( 2 );

			$pdf->SetFont( $font, '', 6 );
			$pdf->SetTextColor( 160, 160, 170 );
			$pdf->MultiCell( $w, 2.8, self::utf8( $d['auto_footer'] ), 0, 'C' );
		}
	}

	// ──────────────────────────────────────────────
	//  HELPERS
	// ──────────────────────────────────────────────

	private static function get_vat_mention( string $regime ): string {
		$mentions = array(
			'franchise'  => 'TVA non applicable, art. 293 B du CGI',
			'autoliquid' => 'Autoliquidation - TVA due par le preneur',
			'intra'      => 'Exoneration TVA, art. 262 ter-I du CGI',
			'export'     => 'Exoneration TVA, art. 262-I du CGI',
		);
		return $mentions[ $regime ] ?? '';
	}

	private static function get_delay_label( string $delay ): string {
		$labels = array(
			'immediate' => 'Paiement comptant a reception',
			'30'        => '30 jours date de facture',
			'30_eom'    => '30 jours fin de mois',
			'45_eom'    => '45 jours fin de mois',
			'60'        => '60 jours date de facture',
		);
		return $labels[ $delay ] ?? '30 jours date de facture';
	}

	private static function hex_to_rgb( string $hex ): array {
		$hex = ltrim( $hex, '#' );
		if ( strlen( $hex ) === 3 ) {
			$hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
		}
		return array(
			(int) hexdec( substr( $hex, 0, 2 ) ),
			(int) hexdec( substr( $hex, 2, 2 ) ),
			(int) hexdec( substr( $hex, 4, 2 ) ),
		);
	}

	private static function map_font( string $font ): string {
		$map = array(
			'Helvetica Neue'  => 'Helvetica',
			'Arial'           => 'Helvetica',
			'Georgia'         => 'Times',
			'Times New Roman' => 'Times',
			'Courier New'     => 'Courier',
		);
		return $map[ $font ] ?? 'Helvetica';
	}

	private static function fmt( float $amount ): string {
		return number_format( $amount, 2, ',', ' ' ) . ' ' . chr( 128 );
	}

	private static function utf8( string $str ): string {
		$result = @iconv( 'UTF-8', 'Windows-1252//TRANSLIT', $str );
		return ( false !== $result ) ? $result : mb_convert_encoding( $str, 'Windows-1252', 'UTF-8' );
	}
}
