<?php
defined( 'ABSPATH' ) || exit;

/**
 * WP REST API endpoints for external invoice access.
 */
class WCEF_REST_API {

	const API_NAMESPACE = 'wcef/v1';

	/**
	 * Register REST routes.
	 */
	public static function init(): void {
		add_action( 'rest_api_init', array( __CLASS__, 'register_routes' ) );
	}

	public static function register_routes(): void {
		register_rest_route( self::API_NAMESPACE, '/invoices', array(
			'methods'             => 'GET',
			'callback'            => array( __CLASS__, 'get_invoices' ),
			'permission_callback' => array( __CLASS__, 'check_permission' ),
			'args'                => array(
				'per_page' => array( 'type' => 'integer', 'default' => 20, 'maximum' => 100, 'sanitize_callback' => 'absint' ),
				'page'     => array( 'type' => 'integer', 'default' => 1, 'sanitize_callback' => 'absint' ),
				'status'   => array( 'type' => 'string', 'default' => '', 'sanitize_callback' => 'sanitize_text_field' ),
				'search'   => array( 'type' => 'string', 'default' => '', 'sanitize_callback' => 'sanitize_text_field' ),
			),
		) );

		register_rest_route( self::API_NAMESPACE, '/invoices/(?P<id>\d+)', array(
			'methods'             => 'GET',
			'callback'            => array( __CLASS__, 'get_invoice' ),
			'permission_callback' => array( __CLASS__, 'check_permission' ),
		) );
	}

	/**
	 * Check permission via API key header or WP cookie auth.
	 */
	public static function check_permission( WP_REST_Request $request ): bool {
		$api_key = $request->get_header( 'X-WCEF-API-Key' );
		if ( $api_key ) {
			$stored_key = get_option( 'wcef_api_key', '' );
			return ! empty( $stored_key ) && hash_equals( $stored_key, $api_key );
		}
		return current_user_can( 'manage_woocommerce' );
	}

	public static function get_invoices( WP_REST_Request $request ): WP_REST_Response {
		if ( ! WCEF_License::can( 'api_webhooks' ) ) {
			return new WP_REST_Response( array( 'error' => 'Feature not available on your plan.' ), 403 );
		}

		$page     = $request->get_param( 'page' );
		$per_page = $request->get_param( 'per_page' );
		$result   = WCEF_Invoice::list( array(
			'per_page' => $per_page,
			'offset'   => ( $page - 1 ) * $per_page,
			'status'   => $request->get_param( 'status' ),
			'search'   => $request->get_param( 'search' ),
		) );

		$response = new WP_REST_Response( array(
			'invoices' => array_map( array( __CLASS__, 'format_invoice' ), $result['items'] ),
			'total'    => $result['total'],
			'page'     => $page,
			'per_page' => $per_page,
		) );
		$response->header( 'X-WP-Total', $result['total'] );
		return $response;
	}

	public static function get_invoice( WP_REST_Request $request ): WP_REST_Response {
		if ( ! WCEF_License::can( 'api_webhooks' ) ) {
			return new WP_REST_Response( array( 'error' => 'Feature not available on your plan.' ), 403 );
		}

		$invoice = WCEF_Invoice::get( (int) $request['id'] );
		if ( ! $invoice ) {
			return new WP_REST_Response( array( 'error' => 'Invoice not found.' ), 404 );
		}

		return new WP_REST_Response( self::format_invoice( $invoice ) );
	}

	private static function format_invoice( object $invoice ): array {
		return array(
			'id'              => (int) $invoice->id,
			'order_id'        => (int) $invoice->order_id,
			'invoice_number'  => $invoice->invoice_number,
			'invoice_type'    => $invoice->invoice_type,
			'status'          => $invoice->status,
			'buyer_siren'     => $invoice->buyer_siren,
			'buyer_tva'       => $invoice->buyer_tva,
			'total_ht'        => (float) $invoice->total_ht,
			'total_tax'       => (float) $invoice->total_tax,
			'total_ttc'       => (float) $invoice->total_ttc,
			'chorus_id'       => $invoice->chorus_id ?? '',
			'transmitted_at'  => $invoice->transmitted_at ?? null,
			'created_at'      => $invoice->created_at,
		);
	}
}
