<?php
defined( 'ABSPATH' ) || exit;

/**
 * Settings helper — provides field definitions and save logic.
 * Settings are rendered via templates/settings-page.php.
 */
class WCEF_Settings {

	/**
	 * All saveable fields with their defaults.
	 */
	public static function get_saveable_fields(): array {
		return array(
			// Company.
			'wcef_company_name'          => get_bloginfo( 'name' ),
			'wcef_siren'                 => '',
			'wcef_siret'                 => '',
			'wcef_tva_intra'             => '',
			'wcef_legal_form'            => '',
			'wcef_address'               => '',
			'wcef_postcode'              => '',
			'wcef_city'                  => '',

			// Legal.
			'wcef_rcs'                   => '',
			'wcef_capital'               => '',
			'wcef_vat_regime'            => 'standard',
			'wcef_option_debits'         => 'no',
			'wcef_payment_delay'         => '30',
			'wcef_penalty_rate'          => "3x le taux d'interet legal",
			'wcef_early_payment'         => "Pas d'escompte pour paiement anticipe",
			'wcef_legal_mentions'        => '',

			// Invoice.
			'wcef_invoice_prefix'        => 'FA-',
			'wcef_next_invoice_number'   => '1',
			'wcef_auto_generate'         => 'yes',
			'wcef_invoice_format'        => 'facturx',

			// Design.
			'wcef_design_show_logo'      => 'yes',
			'wcef_logo_id'               => '0',
			'wcef_design_logo_max_height' => '18',
			'wcef_design_accent'         => '#6366f1',
			'wcef_design_font'           => 'Helvetica Neue',
			'wcef_design_show_footer'    => 'yes',
			'wcef_bank_iban'             => 'FR76 3000 4000 0500 0000 1234 567',
			'wcef_bank_bic'              => 'BNPAFRPPXXX',
			'wcef_bank_name'             => '',
			'wcef_footer_extra'          => '',
		);
	}

	/**
	 * Get a setting value.
	 */
	public static function get( string $key ): string {
		$fields = self::get_saveable_fields();
		return get_option( $key, $fields[ $key ] ?? '' );
	}

	/**
	 * Legal form options.
	 */
	public static function get_legal_forms(): array {
		return array(
			''     => '-- Choisir --',
			'SAS'  => 'SAS',
			'SASU' => 'SASU',
			'SARL' => 'SARL',
			'EURL' => 'EURL',
			'SA'   => 'SA',
			'SCI'  => 'SCI',
			'SNC'  => 'SNC',
			'EI'   => 'Entreprise Individuelle',
			'AE'   => 'Auto-Entrepreneur / Micro-Entreprise',
		);
	}

	/**
	 * VAT regime options.
	 */
	public static function get_vat_regimes(): array {
		return array(
			'standard'   => 'TVA standard (taux normal)',
			'franchise'  => 'Franchise en base de TVA (art. 293 B du CGI)',
			'autoliquid' => 'Autoliquidation',
			'intra'      => 'Livraison intracommunautaire exoneree (art. 262 ter-I)',
			'export'     => 'Exportation hors UE (art. 262-I du CGI)',
		);
	}

	/**
	 * Payment delay options.
	 */
	public static function get_payment_delays(): array {
		return array(
			'immediate' => 'Paiement comptant',
			'30'        => '30 jours',
			'30_eom'    => '30 jours fin de mois',
			'45_eom'    => '45 jours fin de mois',
			'60'        => '60 jours',
		);
	}

	/**
	 * Invoice format options.
	 */
	public static function get_formats(): array {
		return array(
			'facturx' => 'Factur-X (PDF + XML)',
			'cii'     => 'CII (XML seul)',
			'ubl'     => 'UBL (XML seul)',
		);
	}

	/**
	 * Font options.
	 */
	public static function get_fonts(): array {
		return array(
			'Helvetica Neue'  => 'Helvetica Neue',
			'Arial'           => 'Arial',
			'Georgia'         => 'Georgia',
			'Times New Roman' => 'Times New Roman',
			'Courier New'     => 'Courier New',
		);
	}

	/**
	 * Get recommendations based on legal form.
	 */
	public static function get_recommendations( string $legal_form ): array {
		$recs = array();

		// Micro/AE doesn't need RCS or capital — uses franchise TVA.
		if ( in_array( $legal_form, array( 'AE', 'EI' ), true ) ) {
			$recs['vat_regime']     = 'franchise';
			$recs['vat_regime_tip'] = 'Les auto-entrepreneurs et EI sont generalement en franchise en base de TVA (sauf depassement de seuils).';
			$recs['rcs_tip']        = 'Pas de RCS requis pour les EI/auto-entrepreneurs (sauf si commerçant inscrit au RCS).';
			$recs['capital_tip']    = 'Pas de capital social pour les EI/auto-entrepreneurs.';
		}

		// SARL/EURL/SAS/SASU — need RCS and capital.
		if ( in_array( $legal_form, array( 'SAS', 'SASU', 'SARL', 'EURL', 'SA', 'SNC' ), true ) ) {
			$recs['vat_regime']     = 'standard';
			$recs['vat_regime_tip'] = 'Les societes commerciales sont generalement soumises a la TVA au taux normal.';
			$recs['rcs_tip']        = 'Obligatoire — indiquez la ville de votre tribunal de commerce (ex: RCS Paris).';
			$recs['capital_tip']    = 'Obligatoire — indiquez le montant (ex: 10 000 EUR).';
		}

		// SCI.
		if ( 'SCI' === $legal_form ) {
			$recs['vat_regime']     = 'franchise';
			$recs['vat_regime_tip'] = 'Les SCI sont generalement exonerees de TVA sur les locations nues.';
			$recs['rcs_tip']        = 'Obligatoire si la SCI est immatriculee au RCS.';
		}

		return $recs;
	}
}
