<?php
defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Supplier invoices list table for the admin page.
 */
class WCEF_Supplier_Invoice_List extends WP_List_Table {

	public function __construct() {
		parent::__construct( array(
			'singular' => 'supplier_invoice',
			'plural'   => 'supplier_invoices',
			'ajax'     => false,
		) );
	}

	public function get_columns(): array {
		return array(
			'invoice_number' => __( 'N° Facture', 'wc-efacturation-france' ),
			'supplier_name'  => __( 'Fournisseur', 'wc-efacturation-france' ),
			'supplier_siren' => __( 'SIREN', 'wc-efacturation-france' ),
			'total_ht'       => __( 'Total HT', 'wc-efacturation-france' ),
			'total_ttc'      => __( 'Total TTC', 'wc-efacturation-france' ),
			'status'         => __( 'Statut', 'wc-efacturation-france' ),
			'received_at'    => __( 'Recue le', 'wc-efacturation-france' ),
		);
	}

	public function get_sortable_columns(): array {
		return array(
			'invoice_number' => array( 'invoice_number', false ),
			'supplier_name'  => array( 'supplier_name', false ),
			'total_ttc'      => array( 'total_ttc', false ),
			'received_at'    => array( 'received_at', true ),
		);
	}

	public function prepare_items(): void {
		$per_page = 20;
		$page     = $this->get_pagenum();

		$args = array(
			'per_page' => $per_page,
			'offset'   => ( $page - 1 ) * $per_page,
			'orderby'  => sanitize_text_field( wp_unslash( $_GET['orderby'] ?? 'received_at' ) ), // phpcs:ignore WordPress.Security.NonceVerification
			'order'    => sanitize_text_field( wp_unslash( $_GET['order'] ?? 'DESC' ) ), // phpcs:ignore WordPress.Security.NonceVerification
			'search'   => sanitize_text_field( wp_unslash( $_GET['s'] ?? '' ) ), // phpcs:ignore WordPress.Security.NonceVerification
		);

		$result = WCEF_Supplier_Invoice::list( $args );

		$this->items = $result['items'];
		$this->set_pagination_args( array(
			'total_items' => $result['total'],
			'per_page'    => $per_page,
			'total_pages' => ceil( $result['total'] / $per_page ),
		) );

		$this->_column_headers = array(
			$this->get_columns(),
			array(),
			$this->get_sortable_columns(),
		);
	}

	public function column_default( $item, $column_name ): string {
		return esc_html( $item->$column_name ?? '' );
	}

	public function column_invoice_number( object $item ): string {
		return '<strong>' . esc_html( $item->invoice_number ) . '</strong>';
	}

	public function column_supplier_name( object $item ): string {
		$out = esc_html( $item->supplier_name );
		if ( $item->supplier_siret ) {
			$out .= '<br><small>SIRET: ' . esc_html( $item->supplier_siret ) . '</small>';
		}
		return $out;
	}

	public function column_total_ht( object $item ): string {
		return esc_html( number_format( (float) $item->total_ht, 2, ',', ' ' ) ) . ' &euro;';
	}

	public function column_total_ttc( object $item ): string {
		return '<strong>' . esc_html( number_format( (float) $item->total_ttc, 2, ',', ' ' ) ) . ' &euro;</strong>';
	}

	public function column_status( object $item ): string {
		$labels = array(
			'received' => __( 'Recue', 'wc-efacturation-france' ),
		);
		$label = $labels[ $item->status ] ?? $item->status;
		return '<span class="wcef-badge wcef-badge-generated">' . esc_html( $label ) . '</span>';
	}

	public function column_received_at( object $item ): string {
		return esc_html( date_i18n( 'd/m/Y H:i', strtotime( $item->received_at ) ) );
	}

	public function no_items(): void {
		esc_html_e( 'Aucune facture fournisseur recue. Cliquez "Synchroniser" pour recuperer les factures depuis Chorus Pro.', 'wc-efacturation-france' );
	}
}
