<?php
defined( 'ABSPATH' ) || exit;

/**
 * Supplier invoice model — incoming invoices from Chorus Pro.
 */
class WCEF_Supplier_Invoice {

	/**
	 * Get a supplier invoice by ID.
	 */
	public static function get( int $id ): ?object {
		global $wpdb;
		$table = $wpdb->prefix . 'wcef_supplier_invoices';
		return $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table} WHERE id = %d", $id ) ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * List supplier invoices with pagination.
	 */
	public static function list( array $args = array() ): array {
		global $wpdb;
		$table = $wpdb->prefix . 'wcef_supplier_invoices';

		$defaults = array(
			'status'   => '',
			'per_page' => 20,
			'offset'   => 0,
			'orderby'  => 'received_at',
			'order'    => 'DESC',
			'search'   => '',
		);
		$args = wp_parse_args( $args, $defaults );

		$where  = array( '1=1' );
		$values = array();

		if ( $args['status'] ) {
			$where[]  = 'status = %s';
			$values[] = $args['status'];
		}

		if ( $args['search'] ) {
			$where[]  = '(invoice_number LIKE %s OR supplier_name LIKE %s OR supplier_siren LIKE %s)';
			$like     = '%' . $wpdb->esc_like( $args['search'] ) . '%';
			$values[] = $like;
			$values[] = $like;
			$values[] = $like;
		}

		$where_sql = implode( ' AND ', $where );

		$allowed_orderby = array( 'received_at', 'invoice_number', 'total_ttc', 'supplier_name' );
		$orderby = in_array( $args['orderby'], $allowed_orderby, true ) ? $args['orderby'] : 'received_at';
		$order   = 'ASC' === strtoupper( $args['order'] ) ? 'ASC' : 'DESC';

		$count_sql = "SELECT COUNT(*) FROM {$table} WHERE {$where_sql}";
		if ( $values ) {
			$total = (int) $wpdb->get_var( $wpdb->prepare( $count_sql, ...$values ) ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		} else {
			$total = (int) $wpdb->get_var( $count_sql ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		}

		$query        = "SELECT * FROM {$table} WHERE {$where_sql} ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d";
		$query_values = array_merge( $values, array( (int) $args['per_page'], (int) $args['offset'] ) );
		$items        = $wpdb->get_results( $wpdb->prepare( $query, ...$query_values ) ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared

		return array(
			'items' => $items ?: array(),
			'total' => $total,
		);
	}

	/**
	 * Insert or update a supplier invoice from Chorus Pro data.
	 *
	 * @return int|WP_Error Local ID or error.
	 */
	public static function upsert_from_chorus( array $data ) {
		global $wpdb;
		$table     = $wpdb->prefix . 'wcef_supplier_invoices';
		$chorus_id = sanitize_text_field( $data['identifiantFactureCPP'] ?? '' );

		if ( empty( $chorus_id ) ) {
			return new WP_Error( 'missing_id', 'Missing Chorus Pro identifier.' );
		}

		$existing = $wpdb->get_var( $wpdb->prepare(
			"SELECT id FROM {$table} WHERE chorus_id = %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$chorus_id
		) );

		$row = array(
			'chorus_id'      => $chorus_id,
			'supplier_name'  => sanitize_text_field( $data['fournisseurDesignation'] ?? '' ),
			'supplier_siren' => sanitize_text_field( $data['fournisseurSiren'] ?? '' ),
			'supplier_siret' => sanitize_text_field( $data['fournisseurSiret'] ?? '' ),
			'invoice_number' => sanitize_text_field( $data['numeroFacture'] ?? '' ),
			'invoice_date'   => sanitize_text_field( $data['dateFacture'] ?? '' ),
			'total_ht'       => (float) ( $data['montantHT'] ?? 0 ),
			'total_tax'      => (float) ( $data['montantTVA'] ?? 0 ),
			'total_ttc'      => (float) ( $data['montantTTC'] ?? 0 ),
			'status'         => 'received',
		);

		if ( $existing ) {
			$wpdb->update( $table, $row, array( 'id' => (int) $existing ), null, array( '%d' ) );
			return (int) $existing;
		}

		$row['received_at'] = current_time( 'mysql' );
		$wpdb->insert( $table, $row );
		return $wpdb->insert_id;
	}

	/**
	 * Sync supplier invoices from Chorus Pro.
	 *
	 * @return array{ synced: int, errors: int }|WP_Error
	 */
	public static function sync() {
		$result = WCEF_Chorus_Pro::fetch_received_invoices();
		if ( is_wp_error( $result ) ) {
			return $result;
		}

		$invoices = $result['listeFactures'] ?? array();
		$synced   = 0;
		$errors   = 0;

		foreach ( $invoices as $inv ) {
			$upsert = self::upsert_from_chorus( $inv );
			if ( is_wp_error( $upsert ) ) {
				$errors++;
			} else {
				$synced++;
			}
		}

		return array( 'synced' => $synced, 'errors' => $errors );
	}

	/**
	 * Get stats for display.
	 */
	public static function get_stats(): array {
		global $wpdb;
		$table       = $wpdb->prefix . 'wcef_supplier_invoices';
		$month_start = gmdate( 'Y-m-01 00:00:00' );

		$total      = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$this_month = (int) $wpdb->get_var( $wpdb->prepare(
			"SELECT COUNT(*) FROM {$table} WHERE received_at >= %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$month_start
		) );

		return array(
			'total'      => $total,
			'this_month' => $this_month,
		);
	}
}
