<?php
defined( 'ABSPATH' ) || exit;

/**
 * Tracks monthly transmission count and enforces routing tier limits.
 */
class WCEF_Transmission_Counter {

	/**
	 * Get the count of invoices transmitted this month.
	 */
	public static function get_monthly_count(): int {
		global $wpdb;
		$table       = $wpdb->prefix . 'wcef_invoices';
		$month_start = gmdate( 'Y-m-01 00:00:00' );
		return (int) $wpdb->get_var( $wpdb->prepare(
			"SELECT COUNT(*) FROM {$table} WHERE transmitted_at >= %s AND status IN ('transmitted','accepted','delivered')", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$month_start
		) );
	}

	/**
	 * Get the monthly limit for the current routing tier.
	 * Returns -1 for unlimited.
	 */
	public static function get_monthly_limit(): int {
		$limits = WCEF_License::get_limits();
		return (int) ( $limits['routed_per_month'] ?? 0 );
	}

	/**
	 * Check if transmission is allowed (under limit).
	 */
	public static function can_transmit( int $count = 1 ): bool {
		if ( ! WCEF_License::can( 'routage_pa' ) ) {
			return false;
		}
		$limit = self::get_monthly_limit();
		if ( -1 === $limit ) {
			return true;
		}
		return ( self::get_monthly_count() + $count ) <= $limit;
	}

	/**
	 * Get overage count (how many over the limit).
	 */
	public static function get_overage(): int {
		$limit = self::get_monthly_limit();
		if ( -1 === $limit ) {
			return 0;
		}
		$count = self::get_monthly_count();
		return max( 0, $count - $limit );
	}

	/**
	 * Get usage display data for the UI.
	 */
	public static function get_usage_data(): array {
		$count = self::get_monthly_count();
		$limit = self::get_monthly_limit();
		return array(
			'count'      => $count,
			'limit'      => $limit,
			'remaining'  => ( -1 === $limit ) ? -1 : max( 0, $limit - $count ),
			'overage'    => self::get_overage(),
			'percentage' => ( $limit > 0 ) ? min( 100, round( ( $count / $limit ) * 100 ) ) : 0,
		);
	}
}
