<?php
defined( 'ABSPATH' ) || exit;

/**
 * Generates Factur-X / CII / UBL XML from a WooCommerce order.
 * Includes all mandatory French legal fields for 2026 reform compliance.
 */
class WCEF_XML_Generator {

	/**
	 * Generate XML for an order.
	 *
	 * @param WC_Order $order The WooCommerce order.
	 * @param string   $invoice_number The invoice number.
	 * @param string   $format One of: facturx, cii, ubl.
	 * @return string XML content.
	 */
	public static function generate( WC_Order $order, string $invoice_number, string $format = 'facturx' ): string {
		if ( 'ubl' === $format ) {
			return self::generate_ubl( $order, $invoice_number );
		}
		// Both facturx and cii use CII format.
		return self::generate_cii( $order, $invoice_number );
	}

	/**
	 * Generate CII XML (used by Factur-X and standalone CII).
	 */
	private static function generate_cii( WC_Order $order, string $invoice_number ): string {
		$dom = new DOMDocument( '1.0', 'UTF-8' );
		$dom->formatOutput = true;

		$settings = self::get_settings();

		// Root element.
		$root = $dom->createElementNS(
			'urn:un:unece:uncefact:data:standard:CrossIndustryInvoice:100',
			'rsm:CrossIndustryInvoice'
		);
		$root->setAttributeNS( 'http://www.w3.org/2000/xmlns/', 'xmlns:ram', 'urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100' );
		$root->setAttributeNS( 'http://www.w3.org/2000/xmlns/', 'xmlns:udt', 'urn:un:unece:uncefact:data:standard:UnqualifiedDataType:100' );
		$dom->appendChild( $root );

		// ExchangedDocumentContext.
		$context = $dom->createElement( 'rsm:ExchangedDocumentContext' );
		$guideline = self::el( $dom, 'ram:GuidelineSpecifiedDocumentContextParameter' );
		$guideline->appendChild( self::el( $dom, 'ram:ID', 'urn:factur-x.eu:1p0:en16931' ) );
		$context->appendChild( $guideline );
		$root->appendChild( $context );

		// ExchangedDocument.
		$doc = $dom->createElement( 'rsm:ExchangedDocument' );
		$doc->appendChild( self::el( $dom, 'ram:ID', $invoice_number ) );
		$doc->appendChild( self::el( $dom, 'ram:TypeCode', '380' ) ); // Invoice.
		$issue_dt = self::el( $dom, 'ram:IssueDateTime' );
		$dt_str = self::el( $dom, 'udt:DateTimeString', $order->get_date_created()->format( 'Ymd' ) );
		$dt_str->setAttribute( 'format', '102' );
		$issue_dt->appendChild( $dt_str );
		$doc->appendChild( $issue_dt );

		// Include notes for VAT regime and legal mentions.
		$vat_mention = self::get_vat_mention( $settings['vat_regime'] );
		if ( $vat_mention ) {
			$note = self::el( $dom, 'ram:IncludedNote' );
			$note->appendChild( self::el( $dom, 'ram:Content', $vat_mention ) );
			$note->appendChild( self::el( $dom, 'ram:SubjectCode', 'AAK' ) ); // Tax declaration.
			$doc->appendChild( $note );
		}

		// Option sur les débits.
		if ( 'yes' === $settings['option_debits'] ) {
			$note_debits = self::el( $dom, 'ram:IncludedNote' );
			$note_debits->appendChild( self::el( $dom, 'ram:Content', 'Option pour le paiement de la TVA d\'apres les debits' ) );
			$note_debits->appendChild( self::el( $dom, 'ram:SubjectCode', 'AAK' ) );
			$doc->appendChild( $note_debits );
		}

		// Custom legal mentions.
		if ( $settings['legal_mentions'] ) {
			$note_legal = self::el( $dom, 'ram:IncludedNote' );
			$note_legal->appendChild( self::el( $dom, 'ram:Content', $settings['legal_mentions'] ) );
			$note_legal->appendChild( self::el( $dom, 'ram:SubjectCode', 'AAI' ) ); // General information.
			$doc->appendChild( $note_legal );
		}

		$root->appendChild( $doc );

		// SupplyChainTradeTransaction.
		$transaction = $dom->createElement( 'rsm:SupplyChainTradeTransaction' );

		// --- Header Trade Agreement (Seller / Buyer) ---
		$agreement = self::el( $dom, 'ram:ApplicableHeaderTradeAgreement' );

		// Purchase order reference.
		$po_number = $order->get_meta( '_po_number' );
		if ( $po_number ) {
			$po_ref = self::el( $dom, 'ram:BuyerOrderReferencedDocument' );
			$po_ref->appendChild( self::el( $dom, 'ram:IssuerAssignedID', $po_number ) );
			$agreement->appendChild( $po_ref );
		}

		// Seller.
		$seller = self::el( $dom, 'ram:SellerTradeParty' );
		$seller->appendChild( self::el( $dom, 'ram:Name', $settings['company_name'] ) );

		// Seller legal description (legal form + capital + RCS).
		$legal_desc_parts = array();
		if ( $settings['legal_form'] ) {
			$legal_desc_parts[] = $settings['legal_form'];
		}
		if ( $settings['capital'] ) {
			$legal_desc_parts[] = 'Capital : ' . $settings['capital'];
		}
		if ( $settings['rcs'] ) {
			$legal_desc_parts[] = $settings['rcs'];
		}
		if ( $legal_desc_parts ) {
			$seller->appendChild( self::el( $dom, 'ram:Description', implode( ' - ', $legal_desc_parts ) ) );
		}

		$seller_addr = self::el( $dom, 'ram:PostalTradeAddress' );
		$seller_addr->appendChild( self::el( $dom, 'ram:PostcodeCode', $settings['postcode'] ) );
		$seller_addr->appendChild( self::el( $dom, 'ram:LineOne', $settings['address'] ) );
		$seller_addr->appendChild( self::el( $dom, 'ram:CityName', $settings['city'] ) );
		$seller_addr->appendChild( self::el( $dom, 'ram:CountryID', 'FR' ) );
		$seller->appendChild( $seller_addr );

		// Seller SIREN.
		if ( $settings['siren'] ) {
			$seller_legal = self::el( $dom, 'ram:SpecifiedLegalOrganization' );
			$siren_el = self::el( $dom, 'ram:ID', $settings['siren'] );
			$siren_el->setAttribute( 'schemeID', '0002' );
			$seller_legal->appendChild( $siren_el );
			$seller->appendChild( $seller_legal );
		}

		// Seller SIRET.
		if ( $settings['siret'] ) {
			$seller_siret_reg = self::el( $dom, 'ram:SpecifiedTaxRegistration' );
			$siret_el = self::el( $dom, 'ram:ID', $settings['siret'] );
			$siret_el->setAttribute( 'schemeID', '0009' ); // SIRET scheme.
			$seller_siret_reg->appendChild( $siret_el );
			$seller->appendChild( $seller_siret_reg );
		}

		// Seller TVA intra.
		if ( $settings['tva_intra'] && 'franchise' !== $settings['vat_regime'] ) {
			$seller_tax = self::el( $dom, 'ram:SpecifiedTaxRegistration' );
			$tva_id = self::el( $dom, 'ram:ID', $settings['tva_intra'] );
			$tva_id->setAttribute( 'schemeID', 'VA' );
			$seller_tax->appendChild( $tva_id );
			$seller->appendChild( $seller_tax );
		}

		$agreement->appendChild( $seller );

		// Buyer.
		$buyer = self::el( $dom, 'ram:BuyerTradeParty' );
		$buyer_name = $order->get_billing_company() ?: ( $order->get_billing_first_name() . ' ' . $order->get_billing_last_name() );
		$buyer->appendChild( self::el( $dom, 'ram:Name', $buyer_name ) );

		$buyer_addr = self::el( $dom, 'ram:PostalTradeAddress' );
		$buyer_addr->appendChild( self::el( $dom, 'ram:PostcodeCode', $order->get_billing_postcode() ) );
		$buyer_addr->appendChild( self::el( $dom, 'ram:LineOne', $order->get_billing_address_1() ) );
		$buyer_addr->appendChild( self::el( $dom, 'ram:CityName', $order->get_billing_city() ) );
		$buyer_addr->appendChild( self::el( $dom, 'ram:CountryID', $order->get_billing_country() ?: 'FR' ) );
		$buyer->appendChild( $buyer_addr );

		// Buyer SIREN (2026 reform: mandatory for B2B).
		$buyer_siren = $order->get_meta( '_billing_siren' );
		if ( $buyer_siren ) {
			$buyer_legal = self::el( $dom, 'ram:SpecifiedLegalOrganization' );
			$bsiren_el = self::el( $dom, 'ram:ID', $buyer_siren );
			$bsiren_el->setAttribute( 'schemeID', '0002' );
			$buyer_legal->appendChild( $bsiren_el );
			$buyer->appendChild( $buyer_legal );
		}

		// Buyer SIRET.
		$buyer_siret = $order->get_meta( '_billing_siret' );
		if ( $buyer_siret ) {
			$buyer_siret_reg = self::el( $dom, 'ram:SpecifiedTaxRegistration' );
			$bsiret_el = self::el( $dom, 'ram:ID', $buyer_siret );
			$bsiret_el->setAttribute( 'schemeID', '0009' );
			$buyer_siret_reg->appendChild( $bsiret_el );
			$buyer->appendChild( $buyer_siret_reg );
		}

		// Buyer TVA intra.
		$buyer_tva = $order->get_meta( '_billing_tva_intra' );
		if ( $buyer_tva ) {
			$buyer_tax = self::el( $dom, 'ram:SpecifiedTaxRegistration' );
			$btva_id = self::el( $dom, 'ram:ID', $buyer_tva );
			$btva_id->setAttribute( 'schemeID', 'VA' );
			$buyer_tax->appendChild( $btva_id );
			$buyer->appendChild( $buyer_tax );
		}

		$agreement->appendChild( $buyer );
		$transaction->appendChild( $agreement );

		// --- Header Trade Delivery ---
		$delivery = self::el( $dom, 'ram:ApplicableHeaderTradeDelivery' );

		// Delivery address (if different from billing).
		if ( $order->has_shipping_address() ) {
			$ship_to = self::el( $dom, 'ram:ShipToTradeParty' );
			$ship_addr = self::el( $dom, 'ram:PostalTradeAddress' );
			$ship_addr->appendChild( self::el( $dom, 'ram:PostcodeCode', $order->get_shipping_postcode() ) );
			$ship_addr->appendChild( self::el( $dom, 'ram:LineOne', $order->get_shipping_address_1() ) );
			$ship_addr->appendChild( self::el( $dom, 'ram:CityName', $order->get_shipping_city() ) );
			$ship_addr->appendChild( self::el( $dom, 'ram:CountryID', $order->get_shipping_country() ?: 'FR' ) );
			$ship_to->appendChild( $ship_addr );
			$delivery->appendChild( $ship_to );
		}

		$transaction->appendChild( $delivery );

		// --- Header Trade Settlement ---
		$settlement = self::el( $dom, 'ram:ApplicableHeaderTradeSettlement' );
		$settlement->appendChild( self::el( $dom, 'ram:InvoiceCurrencyCode', $order->get_currency() ) );

		// Payment terms.
		$payment_terms = self::el( $dom, 'ram:SpecifiedTradePaymentTerms' );
		$delay_label = self::get_delay_label( $settings['payment_delay'] );
		$description_parts = array( 'Echeance : ' . $delay_label );
		if ( $settings['early_payment'] ) {
			$description_parts[] = 'Escompte : ' . $settings['early_payment'];
		}
		if ( $settings['penalty_rate'] ) {
			$description_parts[] = 'Penalites de retard : ' . $settings['penalty_rate'];
		}
		$description_parts[] = 'Indemnite forfaitaire pour frais de recouvrement : 40,00 EUR';
		$payment_terms->appendChild( self::el( $dom, 'ram:Description', implode( '. ', $description_parts ) ) );

		// Due date based on payment delay.
		$due_date = self::calculate_due_date( $order->get_date_created(), $settings['payment_delay'] );
		if ( $due_date ) {
			$due_dt = self::el( $dom, 'ram:DueDateDateTime' );
			$due_dt_str = self::el( $dom, 'udt:DateTimeString', $due_date->format( 'Ymd' ) );
			$due_dt_str->setAttribute( 'format', '102' );
			$due_dt->appendChild( $due_dt_str );
			$payment_terms->appendChild( $due_dt );
		}
		$settlement->appendChild( $payment_terms );

		// Tax breakdown.
		$tax_totals   = self::get_tax_breakdown( $order );
		$vat_regime   = $settings['vat_regime'];
		$tax_category = self::get_tax_category_code( $vat_regime );

		foreach ( $tax_totals as $rate => $amounts ) {
			$tax = self::el( $dom, 'ram:ApplicableTradeTax' );
			$tax->appendChild( self::el( $dom, 'ram:CalculatedAmount', number_format( $amounts['tax'], 2, '.', '' ) ) );
			$tax->appendChild( self::el( $dom, 'ram:TypeCode', 'VAT' ) );
			$tax->appendChild( self::el( $dom, 'ram:BasisAmount', number_format( $amounts['base'], 2, '.', '' ) ) );
			$tax->appendChild( self::el( $dom, 'ram:CategoryCode', $tax_category ) );
			$tax->appendChild( self::el( $dom, 'ram:RateApplicablePercent', number_format( (float) $rate, 2, '.', '' ) ) );

			// Add exemption reason for non-standard regimes.
			$exemption = self::get_vat_exemption_reason( $vat_regime );
			if ( $exemption ) {
				$tax->appendChild( self::el( $dom, 'ram:ExemptionReason', $exemption ) );
			}

			$settlement->appendChild( $tax );
		}

		// Monetary totals.
		$totals = self::el( $dom, 'ram:SpecifiedTradeSettlementHeaderMonetarySummation' );
		$totals->appendChild( self::el( $dom, 'ram:LineTotalAmount', number_format( (float) $order->get_subtotal(), 2, '.', '' ) ) );
		$totals->appendChild( self::el( $dom, 'ram:TaxBasisTotalAmount', number_format( (float) $order->get_subtotal() + (float) $order->get_shipping_total(), 2, '.', '' ) ) );

		$tax_total_el = self::el( $dom, 'ram:TaxTotalAmount', number_format( (float) $order->get_total_tax(), 2, '.', '' ) );
		$tax_total_el->setAttribute( 'currencyID', $order->get_currency() );
		$totals->appendChild( $tax_total_el );

		$totals->appendChild( self::el( $dom, 'ram:GrandTotalAmount', number_format( (float) $order->get_total(), 2, '.', '' ) ) );
		$totals->appendChild( self::el( $dom, 'ram:DuePayableAmount', number_format( (float) $order->get_total(), 2, '.', '' ) ) );
		$settlement->appendChild( $totals );

		$transaction->appendChild( $settlement );

		// --- Line items ---
		$line_num = 0;
		foreach ( $order->get_items() as $item ) {
			$line_num++;
			$line = self::el( $dom, 'ram:IncludedSupplyChainTradeLineItem' );

			$line_doc = self::el( $dom, 'ram:AssociatedDocumentLineDocument' );
			$line_doc->appendChild( self::el( $dom, 'ram:LineID', (string) $line_num ) );
			$line->appendChild( $line_doc );

			// Product info.
			$product = $item->get_product();
			$trade_product = self::el( $dom, 'ram:SpecifiedTradeProduct' );
			if ( $product && $product->get_sku() ) {
				$trade_product->appendChild( self::el( $dom, 'ram:SellerAssignedID', $product->get_sku() ) );
			}
			$trade_product->appendChild( self::el( $dom, 'ram:Name', $item->get_name() ) );
			$line->appendChild( $trade_product );

			// Line agreement (unit price).
			$line_agreement = self::el( $dom, 'ram:SpecifiedLineTradeAgreement' );
			$net_price = self::el( $dom, 'ram:NetPriceProductTradePrice' );
			$unit_price = (float) $item->get_subtotal() / max( 1, $item->get_quantity() );
			$net_price->appendChild( self::el( $dom, 'ram:ChargeAmount', number_format( $unit_price, 2, '.', '' ) ) );
			$line_agreement->appendChild( $net_price );
			$line->appendChild( $line_agreement );

			// Line delivery (quantity).
			$line_delivery = self::el( $dom, 'ram:SpecifiedLineTradeDelivery' );
			$qty_el = self::el( $dom, 'ram:BilledQuantity', (string) $item->get_quantity() );
			$qty_el->setAttribute( 'unitCode', 'C62' ); // "One" (piece).
			$line_delivery->appendChild( $qty_el );
			$line->appendChild( $line_delivery );

			// Line settlement (total + tax).
			$line_settlement = self::el( $dom, 'ram:SpecifiedLineTradeSettlement' );

			$line_tax = self::el( $dom, 'ram:ApplicableTradeTax' );
			$line_tax->appendChild( self::el( $dom, 'ram:TypeCode', 'VAT' ) );
			$line_tax->appendChild( self::el( $dom, 'ram:CategoryCode', $tax_category ) );

			// Get tax rate for this item.
			$item_tax_rate = 0;
			if ( $item->get_subtotal() > 0 ) {
				$item_tax_rate = round( ( (float) $item->get_subtotal_tax() / (float) $item->get_subtotal() ) * 100, 2 );
			}
			$line_tax->appendChild( self::el( $dom, 'ram:RateApplicablePercent', number_format( $item_tax_rate, 2, '.', '' ) ) );
			$line_settlement->appendChild( $line_tax );

			$line_sum = self::el( $dom, 'ram:SpecifiedTradeSettlementLineMonetarySummation' );
			$line_sum->appendChild( self::el( $dom, 'ram:LineTotalAmount', number_format( (float) $item->get_subtotal(), 2, '.', '' ) ) );
			$line_settlement->appendChild( $line_sum );

			$line->appendChild( $line_settlement );
			$transaction->appendChild( $line );
		}

		// Shipping as a charge line.
		$shipping_total = (float) $order->get_shipping_total();
		if ( $shipping_total > 0 ) {
			$line_num++;
			$ship_line = self::el( $dom, 'ram:IncludedSupplyChainTradeLineItem' );

			$ship_doc = self::el( $dom, 'ram:AssociatedDocumentLineDocument' );
			$ship_doc->appendChild( self::el( $dom, 'ram:LineID', (string) $line_num ) );
			$ship_line->appendChild( $ship_doc );

			$ship_product = self::el( $dom, 'ram:SpecifiedTradeProduct' );
			$ship_product->appendChild( self::el( $dom, 'ram:Name', 'Frais de livraison' ) );
			$ship_line->appendChild( $ship_product );

			$ship_agreement = self::el( $dom, 'ram:SpecifiedLineTradeAgreement' );
			$ship_price = self::el( $dom, 'ram:NetPriceProductTradePrice' );
			$ship_price->appendChild( self::el( $dom, 'ram:ChargeAmount', number_format( $shipping_total, 2, '.', '' ) ) );
			$ship_agreement->appendChild( $ship_price );
			$ship_line->appendChild( $ship_agreement );

			$ship_delivery = self::el( $dom, 'ram:SpecifiedLineTradeDelivery' );
			$ship_qty = self::el( $dom, 'ram:BilledQuantity', '1' );
			$ship_qty->setAttribute( 'unitCode', 'C62' );
			$ship_delivery->appendChild( $ship_qty );
			$ship_line->appendChild( $ship_delivery );

			$ship_settlement = self::el( $dom, 'ram:SpecifiedLineTradeSettlement' );
			$ship_tax = self::el( $dom, 'ram:ApplicableTradeTax' );
			$ship_tax->appendChild( self::el( $dom, 'ram:TypeCode', 'VAT' ) );
			$ship_tax->appendChild( self::el( $dom, 'ram:CategoryCode', $tax_category ) );
			$ship_tax_rate = 0;
			if ( $shipping_total > 0 ) {
				$ship_tax_rate = round( ( (float) $order->get_shipping_tax() / $shipping_total ) * 100, 2 );
			}
			$ship_tax->appendChild( self::el( $dom, 'ram:RateApplicablePercent', number_format( $ship_tax_rate, 2, '.', '' ) ) );
			$ship_settlement->appendChild( $ship_tax );

			$ship_sum = self::el( $dom, 'ram:SpecifiedTradeSettlementLineMonetarySummation' );
			$ship_sum->appendChild( self::el( $dom, 'ram:LineTotalAmount', number_format( $shipping_total, 2, '.', '' ) ) );
			$ship_settlement->appendChild( $ship_sum );

			$ship_line->appendChild( $ship_settlement );
			$transaction->appendChild( $ship_line );
		}

		$root->appendChild( $transaction );

		return $dom->saveXML();
	}

	/**
	 * Generate UBL 2.1 XML.
	 */
	private static function generate_ubl( WC_Order $order, string $invoice_number ): string {
		$dom = new DOMDocument( '1.0', 'UTF-8' );
		$dom->formatOutput = true;

		$settings = self::get_settings();

		$root = $dom->createElementNS( 'urn:oasis:names:specification:ubl:schema:xsd:Invoice-2', 'Invoice' );
		$root->setAttributeNS( 'http://www.w3.org/2000/xmlns/', 'xmlns:cac', 'urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2' );
		$root->setAttributeNS( 'http://www.w3.org/2000/xmlns/', 'xmlns:cbc', 'urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2' );
		$dom->appendChild( $root );

		$root->appendChild( self::el( $dom, 'cbc:UBLVersionID', '2.1' ) );
		$root->appendChild( self::el( $dom, 'cbc:CustomizationID', 'urn:cen.eu:en16931:2017' ) );
		$root->appendChild( self::el( $dom, 'cbc:ID', $invoice_number ) );
		$root->appendChild( self::el( $dom, 'cbc:IssueDate', $order->get_date_created()->format( 'Y-m-d' ) ) );
		$root->appendChild( self::el( $dom, 'cbc:InvoiceTypeCode', '380' ) );
		$root->appendChild( self::el( $dom, 'cbc:DocumentCurrencyCode', $order->get_currency() ) );

		// Purchase order reference.
		$po_number = $order->get_meta( '_po_number' );
		if ( $po_number ) {
			$order_ref = $dom->createElement( 'cac:OrderReference' );
			$order_ref->appendChild( self::el( $dom, 'cbc:ID', $po_number ) );
			$root->appendChild( $order_ref );
		}

		// Notes (VAT regime mention, option debits, legal mentions).
		$vat_mention = self::get_vat_mention( $settings['vat_regime'] );
		if ( $vat_mention ) {
			$root->appendChild( self::el( $dom, 'cbc:Note', $vat_mention ) );
		}
		if ( 'yes' === $settings['option_debits'] ) {
			$root->appendChild( self::el( $dom, 'cbc:Note', 'Option pour le paiement de la TVA d\'apres les debits' ) );
		}
		if ( $settings['legal_mentions'] ) {
			$root->appendChild( self::el( $dom, 'cbc:Note', $settings['legal_mentions'] ) );
		}

		// Supplier.
		$supplier = $dom->createElement( 'cac:AccountingSupplierParty' );
		$supplier_party = $dom->createElement( 'cac:Party' );

		$s_addr = $dom->createElement( 'cac:PostalAddress' );
		$s_addr->appendChild( self::el( $dom, 'cbc:StreetName', $settings['address'] ) );
		$s_addr->appendChild( self::el( $dom, 'cbc:CityName', $settings['city'] ) );
		$s_addr->appendChild( self::el( $dom, 'cbc:PostalZone', $settings['postcode'] ) );
		$s_country = $dom->createElement( 'cac:Country' );
		$s_country->appendChild( self::el( $dom, 'cbc:IdentificationCode', 'FR' ) );
		$s_addr->appendChild( $s_country );
		$supplier_party->appendChild( $s_addr );

		// Supplier VAT.
		if ( $settings['tva_intra'] && 'franchise' !== $settings['vat_regime'] ) {
			$s_tax = $dom->createElement( 'cac:PartyTaxScheme' );
			$s_tax->appendChild( self::el( $dom, 'cbc:CompanyID', $settings['tva_intra'] ) );
			$s_tax_scheme = $dom->createElement( 'cac:TaxScheme' );
			$s_tax_scheme->appendChild( self::el( $dom, 'cbc:ID', 'VAT' ) );
			$s_tax->appendChild( $s_tax_scheme );
			$supplier_party->appendChild( $s_tax );
		}

		$s_legal = $dom->createElement( 'cac:PartyLegalEntity' );
		$s_legal->appendChild( self::el( $dom, 'cbc:RegistrationName', $settings['company_name'] ) );
		if ( $settings['siren'] ) {
			$siren_el = self::el( $dom, 'cbc:CompanyID', $settings['siren'] );
			$siren_el->setAttribute( 'schemeID', '0002' );
			$s_legal->appendChild( $siren_el );
		}

		// Legal description.
		$legal_desc_parts = array();
		if ( $settings['legal_form'] ) {
			$legal_desc_parts[] = $settings['legal_form'];
		}
		if ( $settings['capital'] ) {
			$legal_desc_parts[] = 'Capital : ' . $settings['capital'];
		}
		if ( $settings['rcs'] ) {
			$legal_desc_parts[] = $settings['rcs'];
		}
		if ( $legal_desc_parts ) {
			$s_legal->appendChild( self::el( $dom, 'cbc:CompanyLegalForm', implode( ' - ', $legal_desc_parts ) ) );
		}

		$supplier_party->appendChild( $s_legal );
		$supplier->appendChild( $supplier_party );
		$root->appendChild( $supplier );

		// Customer.
		$customer = $dom->createElement( 'cac:AccountingCustomerParty' );
		$customer_party = $dom->createElement( 'cac:Party' );

		$c_addr = $dom->createElement( 'cac:PostalAddress' );
		$c_addr->appendChild( self::el( $dom, 'cbc:StreetName', $order->get_billing_address_1() ) );
		$c_addr->appendChild( self::el( $dom, 'cbc:CityName', $order->get_billing_city() ) );
		$c_addr->appendChild( self::el( $dom, 'cbc:PostalZone', $order->get_billing_postcode() ) );
		$c_country = $dom->createElement( 'cac:Country' );
		$c_country->appendChild( self::el( $dom, 'cbc:IdentificationCode', $order->get_billing_country() ?: 'FR' ) );
		$c_addr->appendChild( $c_country );
		$customer_party->appendChild( $c_addr );

		// Buyer VAT.
		$buyer_tva = $order->get_meta( '_billing_tva_intra' );
		if ( $buyer_tva ) {
			$c_tax = $dom->createElement( 'cac:PartyTaxScheme' );
			$c_tax->appendChild( self::el( $dom, 'cbc:CompanyID', $buyer_tva ) );
			$c_tax_scheme = $dom->createElement( 'cac:TaxScheme' );
			$c_tax_scheme->appendChild( self::el( $dom, 'cbc:ID', 'VAT' ) );
			$c_tax->appendChild( $c_tax_scheme );
			$customer_party->appendChild( $c_tax );
		}

		$buyer_name = $order->get_billing_company() ?: ( $order->get_billing_first_name() . ' ' . $order->get_billing_last_name() );
		$c_legal = $dom->createElement( 'cac:PartyLegalEntity' );
		$c_legal->appendChild( self::el( $dom, 'cbc:RegistrationName', $buyer_name ) );

		// Buyer SIREN (2026 reform).
		$buyer_siren = $order->get_meta( '_billing_siren' );
		if ( $buyer_siren ) {
			$bsiren_el = self::el( $dom, 'cbc:CompanyID', $buyer_siren );
			$bsiren_el->setAttribute( 'schemeID', '0002' );
			$c_legal->appendChild( $bsiren_el );
		}
		$customer_party->appendChild( $c_legal );

		$customer->appendChild( $customer_party );
		$root->appendChild( $customer );

		// Delivery (shipping address).
		if ( $order->has_shipping_address() ) {
			$ubl_delivery = $dom->createElement( 'cac:Delivery' );
			$delivery_location = $dom->createElement( 'cac:DeliveryLocation' );
			$d_addr = $dom->createElement( 'cac:Address' );
			$d_addr->appendChild( self::el( $dom, 'cbc:StreetName', $order->get_shipping_address_1() ) );
			$d_addr->appendChild( self::el( $dom, 'cbc:CityName', $order->get_shipping_city() ) );
			$d_addr->appendChild( self::el( $dom, 'cbc:PostalZone', $order->get_shipping_postcode() ) );
			$d_country = $dom->createElement( 'cac:Country' );
			$d_country->appendChild( self::el( $dom, 'cbc:IdentificationCode', $order->get_shipping_country() ?: 'FR' ) );
			$d_addr->appendChild( $d_country );
			$delivery_location->appendChild( $d_addr );
			$ubl_delivery->appendChild( $delivery_location );
			$root->appendChild( $ubl_delivery );
		}

		// Payment terms.
		$payment_terms = $dom->createElement( 'cac:PaymentTerms' );
		$terms_parts = array();
		$delay_label = self::get_delay_label( $settings['payment_delay'] );
		$terms_parts[] = 'Echeance : ' . $delay_label;
		if ( $settings['early_payment'] ) {
			$terms_parts[] = 'Escompte : ' . $settings['early_payment'];
		}
		if ( $settings['penalty_rate'] ) {
			$terms_parts[] = 'Penalites de retard : ' . $settings['penalty_rate'];
		}
		$terms_parts[] = 'Indemnite forfaitaire pour frais de recouvrement : 40,00 EUR';
		$payment_terms->appendChild( self::el( $dom, 'cbc:Note', implode( '. ', $terms_parts ) ) );
		$root->appendChild( $payment_terms );

		// Tax total.
		$tax_total = $dom->createElement( 'cac:TaxTotal' );
		$tax_amount = self::el( $dom, 'cbc:TaxAmount', number_format( (float) $order->get_total_tax(), 2, '.', '' ) );
		$tax_amount->setAttribute( 'currencyID', $order->get_currency() );
		$tax_total->appendChild( $tax_amount );

		$tax_totals    = self::get_tax_breakdown( $order );
		$vat_regime    = $settings['vat_regime'];
		$tax_category  = self::get_tax_category_code( $vat_regime );

		foreach ( $tax_totals as $rate => $amounts ) {
			$sub = $dom->createElement( 'cac:TaxSubtotal' );
			$taxable = self::el( $dom, 'cbc:TaxableAmount', number_format( $amounts['base'], 2, '.', '' ) );
			$taxable->setAttribute( 'currencyID', $order->get_currency() );
			$sub->appendChild( $taxable );
			$tax_amt = self::el( $dom, 'cbc:TaxAmount', number_format( $amounts['tax'], 2, '.', '' ) );
			$tax_amt->setAttribute( 'currencyID', $order->get_currency() );
			$sub->appendChild( $tax_amt );
			$cat = $dom->createElement( 'cac:TaxCategory' );
			$cat->appendChild( self::el( $dom, 'cbc:ID', $tax_category ) );
			$cat->appendChild( self::el( $dom, 'cbc:Percent', number_format( (float) $rate, 1, '.', '' ) ) );

			// Add exemption reason for non-standard regimes.
			$exemption = self::get_vat_exemption_reason( $vat_regime );
			if ( $exemption ) {
				$cat->appendChild( self::el( $dom, 'cbc:TaxExemptionReason', $exemption ) );
			}

			$scheme = $dom->createElement( 'cac:TaxScheme' );
			$scheme->appendChild( self::el( $dom, 'cbc:ID', 'VAT' ) );
			$cat->appendChild( $scheme );
			$sub->appendChild( $cat );
			$tax_total->appendChild( $sub );
		}
		$root->appendChild( $tax_total );

		// Legal monetary total.
		$monetary = $dom->createElement( 'cac:LegalMonetaryTotal' );
		$le = self::el( $dom, 'cbc:LineExtensionAmount', number_format( (float) $order->get_subtotal(), 2, '.', '' ) );
		$le->setAttribute( 'currencyID', $order->get_currency() );
		$monetary->appendChild( $le );
		$te = self::el( $dom, 'cbc:TaxExclusiveAmount', number_format( (float) $order->get_subtotal() + (float) $order->get_shipping_total(), 2, '.', '' ) );
		$te->setAttribute( 'currencyID', $order->get_currency() );
		$monetary->appendChild( $te );
		$ti = self::el( $dom, 'cbc:TaxInclusiveAmount', number_format( (float) $order->get_total(), 2, '.', '' ) );
		$ti->setAttribute( 'currencyID', $order->get_currency() );
		$monetary->appendChild( $ti );
		$pa = self::el( $dom, 'cbc:PayableAmount', number_format( (float) $order->get_total(), 2, '.', '' ) );
		$pa->setAttribute( 'currencyID', $order->get_currency() );
		$monetary->appendChild( $pa );
		$root->appendChild( $monetary );

		// Invoice lines.
		$line_num = 0;
		foreach ( $order->get_items() as $item ) {
			$line_num++;
			$inv_line = $dom->createElement( 'cac:InvoiceLine' );
			$inv_line->appendChild( self::el( $dom, 'cbc:ID', (string) $line_num ) );
			$qty = self::el( $dom, 'cbc:InvoicedQuantity', (string) $item->get_quantity() );
			$qty->setAttribute( 'unitCode', 'C62' );
			$inv_line->appendChild( $qty );
			$lea = self::el( $dom, 'cbc:LineExtensionAmount', number_format( (float) $item->get_subtotal(), 2, '.', '' ) );
			$lea->setAttribute( 'currencyID', $order->get_currency() );
			$inv_line->appendChild( $lea );

			$inv_item = $dom->createElement( 'cac:Item' );
			$inv_item->appendChild( self::el( $dom, 'cbc:Name', $item->get_name() ) );

			$item_tax_rate = 0;
			if ( $item->get_subtotal() > 0 ) {
				$item_tax_rate = round( ( (float) $item->get_subtotal_tax() / (float) $item->get_subtotal() ) * 100, 2 );
			}
			$ct = $dom->createElement( 'cac:ClassifiedTaxCategory' );
			$ct->appendChild( self::el( $dom, 'cbc:ID', $tax_category ) );
			$ct->appendChild( self::el( $dom, 'cbc:Percent', number_format( $item_tax_rate, 1, '.', '' ) ) );
			$ts = $dom->createElement( 'cac:TaxScheme' );
			$ts->appendChild( self::el( $dom, 'cbc:ID', 'VAT' ) );
			$ct->appendChild( $ts );
			$inv_item->appendChild( $ct );
			$inv_line->appendChild( $inv_item );

			$price = $dom->createElement( 'cac:Price' );
			$unit_price = (float) $item->get_subtotal() / max( 1, $item->get_quantity() );
			$pa_el = self::el( $dom, 'cbc:PriceAmount', number_format( $unit_price, 2, '.', '' ) );
			$pa_el->setAttribute( 'currencyID', $order->get_currency() );
			$price->appendChild( $pa_el );
			$inv_line->appendChild( $price );

			$root->appendChild( $inv_line );
		}

		return $dom->saveXML();
	}

	/**
	 * Get all plugin settings as an array.
	 */
	private static function get_settings(): array {
		return array(
			'company_name'   => get_option( 'wcef_company_name', get_bloginfo( 'name' ) ),
			'siren'          => get_option( 'wcef_siren', '' ),
			'siret'          => get_option( 'wcef_siret', '' ),
			'tva_intra'      => get_option( 'wcef_tva_intra', '' ),
			'legal_form'     => get_option( 'wcef_legal_form', '' ),
			'address'        => get_option( 'wcef_address', '' ),
			'postcode'       => get_option( 'wcef_postcode', '' ),
			'city'           => get_option( 'wcef_city', '' ),
			'rcs'            => get_option( 'wcef_rcs', '' ),
			'capital'        => get_option( 'wcef_capital', '' ),
			'vat_regime'     => get_option( 'wcef_vat_regime', 'standard' ),
			'option_debits'  => get_option( 'wcef_option_debits', 'no' ),
			'payment_delay'  => get_option( 'wcef_payment_delay', '30' ),
			'penalty_rate'   => get_option( 'wcef_penalty_rate', '3x le taux d\'interet legal' ),
			'early_payment'  => get_option( 'wcef_early_payment', 'Pas d\'escompte pour paiement anticipe' ),
			'legal_mentions' => get_option( 'wcef_legal_mentions', '' ),
		);
	}

	/**
	 * Get VAT regime mention text.
	 */
	private static function get_vat_mention( string $regime ): string {
		$mentions = array(
			'franchise'  => 'TVA non applicable, art. 293 B du CGI',
			'autoliquid' => 'Autoliquidation - TVA due par le preneur',
			'intra'      => 'Exoneration TVA, art. 262 ter-I du CGI',
			'export'     => 'Exoneration TVA, art. 262-I du CGI',
		);
		return $mentions[ $regime ] ?? '';
	}

	/**
	 * Get VAT category code based on regime.
	 */
	private static function get_tax_category_code( string $regime ): string {
		$codes = array(
			'standard'    => 'S',   // Standard rate.
			'franchise'   => 'E',   // Exempt.
			'autoliquid'  => 'AE',  // Reverse charge.
			'intra'       => 'K',   // Intra-community.
			'export'      => 'G',   // Export outside EU.
		);
		return $codes[ $regime ] ?? 'S';
	}

	/**
	 * Get VAT exemption reason for non-standard regimes.
	 */
	private static function get_vat_exemption_reason( string $regime ): string {
		$reasons = array(
			'franchise'  => 'TVA non applicable, art. 293 B du CGI',
			'autoliquid' => 'Autoliquidation',
			'intra'      => 'Exoneration TVA, art. 262 ter-I du CGI',
			'export'     => 'Exoneration TVA, art. 262-I du CGI',
		);
		return $reasons[ $regime ] ?? '';
	}

	/**
	 * Get readable label for payment delay.
	 */
	private static function get_delay_label( string $delay ): string {
		$labels = array(
			'immediate' => 'Paiement comptant a reception',
			'30'        => '30 jours date de facture',
			'30_eom'    => '30 jours fin de mois',
			'45_eom'    => '45 jours fin de mois',
			'60'        => '60 jours date de facture',
		);
		return $labels[ $delay ] ?? '30 jours date de facture';
	}

	/**
	 * Calculate due date from order date and payment delay.
	 */
	private static function calculate_due_date( ?WC_DateTime $order_date, string $delay ): ?\DateTime {
		if ( ! $order_date ) {
			return null;
		}

		$date = new \DateTime( $order_date->format( 'Y-m-d' ) );

		switch ( $delay ) {
			case 'immediate':
				return $date;
			case '30':
				$date->modify( '+30 days' );
				return $date;
			case '30_eom':
				$date->modify( '+30 days' );
				$date->modify( 'last day of this month' );
				return $date;
			case '45_eom':
				$date->modify( '+45 days' );
				$date->modify( 'last day of this month' );
				return $date;
			case '60':
				$date->modify( '+60 days' );
				return $date;
			default:
				$date->modify( '+30 days' );
				return $date;
		}
	}

	/**
	 * Create a DOM element with properly escaped text content.
	 * Unlike DOMDocument::createElement($tag, $value), this handles &, <, >, etc.
	 */
	private static function el( DOMDocument $dom, string $tag, string $value = '' ): DOMElement {
		$el = $dom->createElement( $tag );
		if ( '' !== $value ) {
			$el->appendChild( $dom->createTextNode( $value ) );
		}
		return $el;
	}

	/**
	 * Get tax breakdown from order (rate => base + tax amounts).
	 */
	private static function get_tax_breakdown( WC_Order $order ): array {
		$result = array();

		foreach ( $order->get_items( 'tax' ) as $tax_item ) {
			$rate_id    = $tax_item->get_rate_id();
			$rate_pct   = \WC_Tax::_get_tax_rate( $rate_id );
			$rate_value = $rate_pct ? (float) $rate_pct['tax_rate'] : 20.0;
			$key        = (string) $rate_value;

			if ( ! isset( $result[ $key ] ) ) {
				$result[ $key ] = array(
					'base' => 0,
					'tax'  => 0,
				);
			}
			$result[ $key ]['tax'] += (float) $tax_item->get_tax_total() + (float) $tax_item->get_shipping_tax_total();
		}

		// Calculate base from tax and rate.
		foreach ( $result as $rate => &$amounts ) {
			$rate_float = (float) $rate;
			if ( $rate_float > 0 ) {
				$amounts['base'] = round( $amounts['tax'] / ( $rate_float / 100 ), 2 );
			}
		}

		// Fallback if no tax data.
		if ( empty( $result ) ) {
			$result['20'] = array(
				'base' => (float) $order->get_subtotal() + (float) $order->get_shipping_total(),
				'tax'  => (float) $order->get_total_tax(),
			);
		}

		return $result;
	}
}
