<?php
/**
 * Plugin Name:       E-Facturation France
 * Plugin URI:        https://efacturx.com
 * Description:       Facturation electronique conforme pour WooCommerce — generez des factures Factur-X et routez-les vers votre Plateforme Agreee.
 * Version:           1.0.0
 * Requires at least: 6.2
 * Requires PHP:      7.4
 * WC requires at least: 7.0
 * WC tested up to:   9.6
 * Author:            E-Facturx
 * Author URI:        https://efacturx.com
 * License:           GPL-2.0-or-later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       wc-efacturation-france
 * Domain Path:       /languages
 */

defined( 'ABSPATH' ) || exit;

define( 'WCEF_VERSION', '1.0.0' );
define( 'WCEF_PLUGIN_FILE', __FILE__ );
define( 'WCEF_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'WCEF_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

// Load all includes.
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-install.php';
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-settings.php';
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-checkout-fields.php';
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-invoice.php';
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-invoice-list.php';
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-xml-generator.php';
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-pdf-generator.php';
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-license.php';
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-transmission-counter.php';
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-chorus-pro.php';
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-cron.php';
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-supplier-invoice.php';
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-supplier-invoice-list.php';
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-export.php';
require_once WCEF_PLUGIN_DIR . 'includes/class-wcef-rest-api.php';

/**
 * Main plugin class.
 */
final class WC_EFacturation_France {

	private static ?self $instance = null;

	public static function instance(): self {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		$this->init_hooks();
		WCEF_Cron::init();
		WCEF_REST_API::init();

		// Run DB migrations on admin_init.
		add_action( 'admin_init', array( 'WCEF_Install', 'maybe_upgrade' ) );
	}

	private function init_hooks(): void {
		// Admin menu — top-level, right after Payments.
		add_action( 'admin_menu', array( $this, 'admin_menu' ), 30 );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_assets' ) );
		add_action( 'admin_notices', array( $this, 'admin_notices' ) );

		// Checkout B2B fields.
		WCEF_Checkout_Fields::init();

		// Invoice generation on order status change.
		add_action( 'woocommerce_order_status_completed', array( $this, 'maybe_generate_invoice' ) );
		add_action( 'woocommerce_order_status_processing', array( $this, 'maybe_generate_invoice' ) );

		// Meta box on order screen.
		add_action( 'add_meta_boxes', array( $this, 'add_order_meta_box' ) );

		// AJAX handlers.
		add_action( 'wp_ajax_wcef_generate_invoice', array( $this, 'ajax_generate_invoice' ) );
		add_action( 'wp_ajax_wcef_download_pdf', array( $this, 'ajax_download_pdf' ) );
		add_action( 'wp_ajax_wcef_download_xml', array( $this, 'ajax_download_xml' ) );
		add_action( 'wp_ajax_wcef_dismiss_banner', array( $this, 'ajax_dismiss_banner' ) );
		add_action( 'wp_ajax_wcef_save_settings', array( $this, 'ajax_save_settings' ) );
		add_action( 'wp_ajax_wcef_preview_pdf', array( $this, 'ajax_preview_pdf' ) );
		add_action( 'wp_ajax_wcef_verify_license', array( $this, 'ajax_verify_license' ) );
		add_action( 'wp_ajax_wcef_deactivate_license', array( $this, 'ajax_deactivate_license' ) );

		// Chorus Pro / Transmission handlers.
		add_action( 'wp_ajax_wcef_transmit_invoice', array( $this, 'ajax_transmit_invoice' ) );
		add_action( 'wp_ajax_wcef_test_chorus', array( $this, 'ajax_test_chorus' ) );
		add_action( 'wp_ajax_wcef_check_invoice_status', array( $this, 'ajax_check_invoice_status' ) );
		add_action( 'wp_ajax_wcef_bulk_transmit', array( $this, 'ajax_bulk_transmit' ) );
		add_action( 'wp_ajax_wcef_fetch_supplier_invoices', array( $this, 'ajax_fetch_supplier_invoices' ) );
		add_action( 'wp_ajax_wcef_export_csv', array( $this, 'ajax_export_csv' ) );
		add_action( 'wp_ajax_wcef_generate_api_key', array( $this, 'ajax_generate_api_key' ) );

		// Plugin action links.
		add_filter( 'plugin_action_links_' . plugin_basename( WCEF_PLUGIN_FILE ), array( $this, 'action_links' ) );
	}

	/**
	 * Register top-level admin menu below Payments.
	 */
	public function admin_menu(): void {
		// SVG icon for the menu (invoice/document icon).
		$icon = 'data:image/svg+xml;base64,' . base64_encode(
			'<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/><polyline points="14 2 14 8 20 8"/><line x1="8" y1="13" x2="16" y2="13"/><line x1="8" y1="17" x2="13" y2="17"/></svg>'
		);

		// Top-level menu at position 56.5 (right after Payments at 56).
		add_menu_page(
			__( 'E-Facturation', 'wc-efacturation-france' ),
			__( 'E-Facturation', 'wc-efacturation-france' ),
			'manage_woocommerce',
			'wcef-invoices',
			array( $this, 'render_invoices_page' ),
			$icon,
			'56.5'
		);

		// Submenu: Factures (same as parent — WP convention).
		add_submenu_page(
			'wcef-invoices',
			__( 'Factures', 'wc-efacturation-france' ),
			__( 'Factures', 'wc-efacturation-france' ),
			'manage_woocommerce',
			'wcef-invoices',
			array( $this, 'render_invoices_page' )
		);

		// Submenu: Fournisseurs (only if feature is available).
		if ( WCEF_License::can( 'reception_fournisseurs' ) ) {
			add_submenu_page(
				'wcef-invoices',
				__( 'Fournisseurs', 'wc-efacturation-france' ),
				__( 'Fournisseurs', 'wc-efacturation-france' ),
				'manage_woocommerce',
				'wcef-suppliers',
				array( $this, 'render_suppliers_page' )
			);
		}

		// Submenu: Reglages.
		add_submenu_page(
			'wcef-invoices',
			__( 'Reglages', 'wc-efacturation-france' ),
			__( 'Reglages', 'wc-efacturation-france' ),
			'manage_woocommerce',
			'wcef-settings',
			array( $this, 'render_settings_page' )
		);
	}

	/**
	 * Enqueue admin CSS/JS.
	 */
	public function admin_assets( string $hook ): void {
		$screen = get_current_screen();
		if ( ! $screen ) {
			return;
		}

		// Our screens + WooCommerce order screens.
		$our_pages = array(
			'toplevel_page_wcef-invoices',
			'e-facturation_page_wcef-settings',
			'e-facturation_page_wcef-suppliers',
		);
		$is_order = in_array( $screen->id, array( 'shop_order', 'woocommerce_page_wc-orders' ), true );

		if ( in_array( $screen->id, $our_pages, true ) || $is_order || 'plugins' === $screen->id ) {
			wp_enqueue_style( 'wcef-admin', WCEF_PLUGIN_URL . 'admin/css/admin.css', array(), WCEF_VERSION );
			wp_enqueue_script( 'wcef-admin', WCEF_PLUGIN_URL . 'admin/js/admin.js', array( 'jquery' ), WCEF_VERSION, true );
			$license_data = WCEF_License::verify();
			wp_localize_script( 'wcef-admin', 'wcef', array(
				'ajax_url'    => admin_url( 'admin-ajax.php' ),
				'nonce'       => wp_create_nonce( 'wcef_nonce' ),
				'plan'        => $license_data['plan'],
				'routing'     => $license_data['routing'] ?? 'none',
				'has_license' => ! empty( get_option( 'wcef_license_key', '' ) ),
			) );

			// Media library for logo upload on settings page.
			if ( 'e-facturation_page_wcef-settings' === $screen->id ) {
				wp_enqueue_media();
			}
		}
	}

	/**
	 * Render the invoices list page.
	 */
	public function render_invoices_page(): void {
		$list_table = new WCEF_Invoice_List();
		$list_table->prepare_items();
		include WCEF_PLUGIN_DIR . 'templates/invoices-page.php';
	}

	/**
	 * Render the standalone settings page.
	 */
	public function render_settings_page(): void {
		include WCEF_PLUGIN_DIR . 'templates/settings-page.php';
	}

	/**
	 * Render the suppliers page.
	 */
	public function render_suppliers_page(): void {
		$list_table = new WCEF_Supplier_Invoice_List();
		$list_table->prepare_items();
		include WCEF_PLUGIN_DIR . 'templates/suppliers-page.php';
	}

	/**
	 * Auto-generate invoice on order status change.
	 */
	public function maybe_generate_invoice( int $order_id ): void {
		$auto = get_option( 'wcef_auto_generate', 'yes' );
		if ( 'yes' !== $auto ) {
			return;
		}

		$existing = WCEF_Invoice::get_by_order_id( $order_id );
		if ( $existing ) {
			return;
		}

		WCEF_Invoice::generate( $order_id );
	}

	/**
	 * Add meta box on order edit screen.
	 */
	public function add_order_meta_box(): void {
		$screen = wc_get_container()->get( \Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController::class )->custom_orders_table_usage_is_enabled()
			? wc_get_page_screen_id( 'shop-order' )
			: 'shop_order';

		add_meta_box(
			'wcef-invoice-box',
			__( 'E-Facturation', 'wc-efacturation-france' ),
			array( $this, 'render_order_meta_box' ),
			$screen,
			'side',
			'default'
		);
	}

	/**
	 * Render the order meta box content.
	 */
	public function render_order_meta_box( $post_or_order ): void {
		$order = ( $post_or_order instanceof WC_Order ) ? $post_or_order : wc_get_order( $post_or_order->ID );
		if ( ! $order ) {
			return;
		}
		$invoice = WCEF_Invoice::get_by_order_id( $order->get_id() );
		include WCEF_PLUGIN_DIR . 'templates/order-meta-box.php';
	}

	/**
	 * AJAX: Generate invoice.
	 */
	public function ajax_generate_invoice(): void {
		check_ajax_referer( 'wcef_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( 'Permission denied.' );
		}

		$order_id = absint( $_POST['order_id'] ?? 0 );
		if ( ! $order_id ) {
			wp_send_json_error( 'Invalid order ID.' );
		}

		$result = WCEF_Invoice::generate( $order_id );
		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		wp_send_json_success( array( 'invoice_number' => $result ) );
	}

	/**
	 * AJAX: Download PDF.
	 */
	public function ajax_download_pdf(): void {
		check_ajax_referer( 'wcef_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( esc_html__( 'Vous n\'avez pas les droits necessaires.', 'wc-efacturation-france' ), 403 );
		}

		$invoice_id = absint( $_GET['invoice_id'] ?? 0 );
		$invoice    = WCEF_Invoice::get( $invoice_id );
		if ( ! $invoice ) {
			wp_die( esc_html__( 'Facture introuvable.', 'wc-efacturation-france' ), 404 );
		}

		$safe_name = sanitize_file_name( 'facture-' . $invoice->invoice_number . '.pdf' );

		// Serve the immutable PDF saved at generation time.
		if ( ! empty( $invoice->pdf_path ) ) {
			$full_path = WCEF_Invoice::get_pdf_full_path( $invoice->pdf_path );
			if ( file_exists( $full_path ) ) {
				header( 'Content-Type: application/pdf' );
				header( 'Content-Disposition: attachment; filename="' . $safe_name . '"' );
				header( 'Content-Length: ' . filesize( $full_path ) );
				readfile( $full_path ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_readfile
				exit;
			}
		}

		// Fallback for invoices generated before PDF-to-disk was implemented.
		$pdf = WCEF_PDF_Generator::generate( $invoice );

		header( 'Content-Type: application/pdf' );
		header( 'Content-Disposition: attachment; filename="' . $safe_name . '"' );
		header( 'Content-Length: ' . strlen( $pdf ) );
		echo $pdf; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		exit;
	}

	/**
	 * AJAX: Download XML.
	 */
	public function ajax_download_xml(): void {
		check_ajax_referer( 'wcef_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( esc_html__( 'Vous n\'avez pas les droits necessaires.', 'wc-efacturation-france' ), 403 );
		}

		$invoice_id = absint( $_GET['invoice_id'] ?? 0 );
		$invoice    = WCEF_Invoice::get( $invoice_id );
		if ( ! $invoice ) {
			wp_die( esc_html__( 'Facture introuvable.', 'wc-efacturation-france' ), 404 );
		}

		$safe_name = sanitize_file_name( 'facture-' . $invoice->invoice_number . '.xml' );
		header( 'Content-Type: application/xml' );
		header( 'Content-Disposition: attachment; filename="' . $safe_name . '"' );
		header( 'Content-Length: ' . strlen( $invoice->xml_content ) );
		echo $invoice->xml_content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		exit;
	}

	/**
	 * AJAX: Save settings.
	 */
	public function ajax_save_settings(): void {
		check_ajax_referer( 'wcef_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( 'Permission denied.' );
		}

		// Determine which tab is being saved so we only touch its fields.
		$active_tab = sanitize_text_field( wp_unslash( $_POST['wcef_active_tab'] ?? '' ) );

		$tab_fields = array(
			'entreprise'  => array( 'wcef_company_name', 'wcef_legal_form', 'wcef_siren', 'wcef_siret', 'wcef_tva_intra', 'wcef_rcs', 'wcef_capital', 'wcef_address', 'wcef_postcode', 'wcef_city' ),
			'facturation' => array( 'wcef_invoice_prefix', 'wcef_next_invoice_number', 'wcef_invoice_format', 'wcef_auto_generate', 'wcef_design_show_footer', 'wcef_bank_iban', 'wcef_bank_bic', 'wcef_bank_name', 'wcef_footer_extra' ),
			'legal'       => array( 'wcef_vat_regime', 'wcef_option_debits', 'wcef_payment_delay', 'wcef_penalty_rate', 'wcef_early_payment', 'wcef_legal_mentions' ),
			'design'      => array( 'wcef_design_show_logo', 'wcef_logo_id', 'wcef_design_logo_max_height', 'wcef_design_accent', 'wcef_design_font' ),
			'chorus'      => array( 'wcef_chorus_environment', 'wcef_chorus_client_id', 'wcef_chorus_client_secret', 'wcef_chorus_tech_login', 'wcef_chorus_tech_password', 'wcef_chorus_structure_id' ),
		);

		// License key is handled separately via its own AJAX handler, not via the save flow.

		$saveable    = isset( $tab_fields[ $active_tab ] ) ? $tab_fields[ $active_tab ] : array_keys( WCEF_Settings::get_saveable_fields() );
		$checkboxes  = array( 'wcef_auto_generate', 'wcef_design_show_logo', 'wcef_design_show_footer', 'wcef_option_debits' );
		$textareas   = array( 'wcef_address', 'wcef_legal_mentions', 'wcef_footer_extra' );

		foreach ( $saveable as $key ) {
			if ( in_array( $key, $checkboxes, true ) ) {
				// Checkbox: absent from POST = unchecked.
				update_option( $key, isset( $_POST[ $key ] ) ? 'yes' : 'no' );
			} elseif ( in_array( $key, $textareas, true ) ) {
				if ( isset( $_POST[ $key ] ) ) {
					update_option( $key, sanitize_textarea_field( wp_unslash( $_POST[ $key ] ) ) );
				}
			} elseif ( isset( $_POST[ $key ] ) ) {
				update_option( $key, sanitize_text_field( wp_unslash( $_POST[ $key ] ) ) );
			}
		}

		wp_send_json_success( array( 'message' => 'Reglages enregistres.' ) );
	}

	/**
	 * AJAX: Generate a live PDF preview with current (unsaved) design settings.
	 * Streams the PDF inline for iframe display.
	 */
	public function ajax_preview_pdf(): void {
		// Support both GET (iframe src) and POST (AJAX).
		$nonce = sanitize_text_field( wp_unslash( $_REQUEST['nonce'] ?? '' ) );
		if ( ! wp_verify_nonce( $nonce, 'wcef_nonce' ) ) {
			wp_die( 'Nonce invalide.', 403 );
		}

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( 'Permission denied.', 403 );
		}

		// Determine which fields belong to the active tab.
		$active_tab = sanitize_text_field( wp_unslash( $_REQUEST['wcef_active_tab'] ?? '' ) ); // phpcs:ignore WordPress.Security.NonceVerification

		$tab_fields = array(
			'entreprise'  => array( 'wcef_company_name', 'wcef_legal_form', 'wcef_siren', 'wcef_siret', 'wcef_tva_intra', 'wcef_rcs', 'wcef_capital', 'wcef_address', 'wcef_postcode', 'wcef_city' ),
			'facturation' => array( 'wcef_invoice_prefix', 'wcef_next_invoice_number', 'wcef_invoice_format', 'wcef_auto_generate', 'wcef_design_show_footer', 'wcef_bank_iban', 'wcef_bank_bic', 'wcef_bank_name', 'wcef_footer_extra' ),
			'legal'       => array( 'wcef_vat_regime', 'wcef_option_debits', 'wcef_payment_delay', 'wcef_penalty_rate', 'wcef_early_payment', 'wcef_legal_mentions' ),
			'design'      => array( 'wcef_design_show_logo', 'wcef_logo_id', 'wcef_design_logo_max_height', 'wcef_design_accent', 'wcef_design_font' ),
		);

		// Only override fields that belong to the active tab; other tabs keep saved values.
		$fields_to_override = isset( $tab_fields[ $active_tab ] ) ? $tab_fields[ $active_tab ] : array();
		$textareas          = array( 'wcef_address', 'wcef_legal_mentions', 'wcef_footer_extra' );
		$checkboxes         = array( 'wcef_auto_generate', 'wcef_design_show_logo', 'wcef_design_show_footer', 'wcef_option_debits' );

		foreach ( $fields_to_override as $key ) {
			if ( ! isset( $_REQUEST[ $key ] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
				// Checkbox absent from POST = unchecked.
				if ( in_array( $key, $checkboxes, true ) ) {
					$value = 'no';
					add_filter( "pre_option_{$key}", function () use ( $value ) {
						return $value;
					} );
				}
				continue;
			}
			$raw = wp_unslash( $_REQUEST[ $key ] ); // phpcs:ignore WordPress.Security.NonceVerification
			if ( 'wcef_design_accent' === $key ) {
				$value = sanitize_hex_color( $raw ) ?: get_option( $key, '#6366f1' );
			} elseif ( in_array( $key, $textareas, true ) ) {
				$value = sanitize_textarea_field( $raw );
			} elseif ( in_array( $key, array( 'wcef_logo_id', 'wcef_design_logo_max_height', 'wcef_next_invoice_number' ), true ) ) {
				$value = absint( $raw );
			} else {
				$value = sanitize_text_field( $raw );
			}
			add_filter( "pre_option_{$key}", function () use ( $value ) {
				return $value;
			} );
		}

		// Find a real invoice to use for preview.
		global $wpdb;
		$table   = $wpdb->prefix . 'wcef_invoices';
		$invoice = $wpdb->get_row( "SELECT * FROM {$table} ORDER BY id DESC LIMIT 1" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		// If no invoice exists, find any WooCommerce order and build a mock invoice from it.
		if ( ! $invoice ) {
			$orders = wc_get_orders( array( 'limit' => 1, 'orderby' => 'date', 'order' => 'DESC' ) );
			$mock_order_id = ! empty( $orders ) ? $orders[0]->get_id() : 0;

			$invoice = (object) array(
				'id'             => 0,
				'order_id'       => $mock_order_id,
				'invoice_number' => 'FA-000001',
				'invoice_type'   => '380',
				'status'         => 'generated',
				'total_ht'       => 100.00,
				'total_tax'      => 20.00,
				'total_ttc'      => 120.00,
				'created_at'     => current_time( 'mysql' ),
			);
		}

		$pdf_content = WCEF_PDF_Generator::generate( $invoice );

		if ( empty( $pdf_content ) ) {
			wp_die( 'Impossible de generer l\'apercu.', 500 );
		}

		// Stream PDF inline.
		header( 'Content-Type: application/pdf' );
		header( 'Content-Disposition: inline; filename="preview.pdf"' );
		header( 'Content-Length: ' . strlen( $pdf_content ) );
		header( 'Cache-Control: no-cache, no-store, must-revalidate' );
		echo $pdf_content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		exit;
	}

	/**
	 * AJAX: Verify license key.
	 */
	public function ajax_verify_license(): void {
		check_ajax_referer( 'wcef_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( 'Permission denied.' );
		}

		$key = sanitize_text_field( wp_unslash( $_POST['license_key'] ?? '' ) );
		if ( empty( $key ) ) {
			wp_send_json_error( 'Cle de licence requise.' );
		}

		update_option( 'wcef_license_key', $key );
		$result = WCEF_License::refresh();

		if ( ! empty( $result['valid'] ) && 'starter' !== $result['plan'] ) {
			wp_send_json_success( $result );
		} else {
			// Invalid key — remove it.
			delete_option( 'wcef_license_key' );
			delete_transient( WCEF_License::TRANSIENT_KEY );
			wp_send_json_error( $result['error'] ?? 'Licence invalide ou expiree.' );
		}
	}

	/**
	 * AJAX: Deactivate license.
	 */
	public function ajax_deactivate_license(): void {
		check_ajax_referer( 'wcef_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( 'Permission denied.' );
		}

		WCEF_License::deactivate();
		wp_send_json_success( array( 'plan' => 'starter', 'routing' => 'none' ) );
	}

	/**
	 * AJAX: Transmit a single invoice to Chorus Pro.
	 */
	public function ajax_transmit_invoice(): void {
		check_ajax_referer( 'wcef_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( 'Permission denied.' );
		}

		$invoice_id = absint( $_POST['invoice_id'] ?? 0 );
		$invoice    = WCEF_Invoice::get( $invoice_id );
		if ( ! $invoice ) {
			wp_send_json_error( __( 'Facture introuvable.', 'wc-efacturation-france' ) );
		}

		$result = WCEF_Chorus_Pro::submit_invoice( $invoice );
		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		wp_send_json_success( array(
			'message'   => __( 'Facture transmise avec succes.', 'wc-efacturation-france' ),
			'chorus_id' => $result['identifiantFactureCPP'] ?? '',
		) );
	}

	/**
	 * AJAX: Test Chorus Pro connection.
	 */
	public function ajax_test_chorus(): void {
		check_ajax_referer( 'wcef_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( 'Permission denied.' );
		}

		$result = WCEF_Chorus_Pro::test_connection();
		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		wp_send_json_success( array( 'message' => __( 'Connexion reussie.', 'wc-efacturation-france' ) ) );
	}

	/**
	 * AJAX: Check invoice status on Chorus Pro.
	 */
	public function ajax_check_invoice_status(): void {
		check_ajax_referer( 'wcef_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( 'Permission denied.' );
		}

		$invoice_id = absint( $_POST['invoice_id'] ?? 0 );
		$invoice    = WCEF_Invoice::get( $invoice_id );
		if ( ! $invoice ) {
			wp_send_json_error( __( 'Facture introuvable.', 'wc-efacturation-france' ) );
		}

		$result = WCEF_Chorus_Pro::check_status( $invoice );
		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		$updated = WCEF_Invoice::get( $invoice_id );
		wp_send_json_success( array(
			'status'       => $updated->status,
			'status_label' => WCEF_Invoice::get_status_labels()[ $updated->status ] ?? $updated->status,
		) );
	}

	/**
	 * AJAX: Bulk transmit invoices.
	 */
	public function ajax_bulk_transmit(): void {
		check_ajax_referer( 'wcef_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( 'Permission denied.' );
		}

		$ids     = array_map( 'absint', (array) ( $_POST['invoice_ids'] ?? array() ) );
		$success = 0;
		$errors  = 0;
		$messages = array();

		foreach ( $ids as $id ) {
			$invoice = WCEF_Invoice::get( $id );
			if ( ! $invoice || 'generated' !== $invoice->status ) {
				$errors++;
				$messages[] = sprintf( __( 'Facture #%d: statut invalide.', 'wc-efacturation-france' ), $id );
				continue;
			}

			$result = WCEF_Chorus_Pro::submit_invoice( $invoice );
			if ( is_wp_error( $result ) ) {
				$errors++;
				$messages[] = sprintf( __( 'Facture %s: %s', 'wc-efacturation-france' ), $invoice->invoice_number, $result->get_error_message() );
			} else {
				$success++;
			}
		}

		wp_send_json_success( array(
			'success'  => $success,
			'errors'   => $errors,
			'messages' => $messages,
		) );
	}

	/**
	 * AJAX: Sync supplier invoices from Chorus Pro.
	 */
	public function ajax_fetch_supplier_invoices(): void {
		check_ajax_referer( 'wcef_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( 'Permission denied.' );
		}

		$result = WCEF_Supplier_Invoice::sync();
		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		wp_send_json_success( array(
			'message' => sprintf(
				__( '%d factures synchronisees, %d erreurs.', 'wc-efacturation-france' ),
				$result['synced'],
				$result['errors']
			),
			'synced' => $result['synced'],
			'errors' => $result['errors'],
		) );
	}

	/**
	 * AJAX: Export CSV.
	 */
	public function ajax_export_csv(): void {
		check_ajax_referer( 'wcef_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( 'Permission denied.', 403 );
		}

		$args = array();
		if ( ! empty( $_GET['status'] ) ) {
			$args['status'] = sanitize_text_field( wp_unslash( $_GET['status'] ) );
		}

		$csv  = WCEF_Export::generate_csv( $args );
		$name = 'factures-' . gmdate( 'Y-m-d' ) . '.csv';

		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename="' . $name . '"' );
		header( 'Content-Length: ' . strlen( $csv ) );
		echo $csv; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		exit;
	}

	/**
	 * AJAX: Generate API key.
	 */
	public function ajax_generate_api_key(): void {
		check_ajax_referer( 'wcef_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( 'Permission denied.' );
		}

		$key = 'wcef_' . wp_generate_password( 32, false );
		update_option( 'wcef_api_key', $key );
		wp_send_json_success( array( 'api_key' => $key ) );
	}

	/**
	 * Show onboarding notice if settings are not configured.
	 */
	public function admin_notices(): void {
		$screen = get_current_screen();
		if ( ! $screen ) {
			return;
		}

		// Only on our plugin pages.
		$allowed = array( 'toplevel_page_wcef-invoices', 'e-facturation_page_wcef-settings', 'plugins' );
		if ( ! in_array( $screen->id, $allowed, true ) ) {
			return;
		}

		$siren = get_option( 'wcef_siren', '' );
		if ( empty( $siren ) ) {
			?>
			<div class="wcef-welcome-notice notice is-dismissible">
				<div class="wcef-notice-icon">&#9889;</div>
				<div class="wcef-notice-content">
					<h3><?php esc_html_e( 'Configurez E-Facturation', 'wc-efacturation-france' ); ?></h3>
					<p><?php esc_html_e( 'Renseignez votre SIREN, SIRET et numero de TVA pour generer des factures conformes.', 'wc-efacturation-france' ); ?></p>
				</div>
				<div class="wcef-notice-actions">
					<a href="<?php echo esc_url( admin_url( 'admin.php?page=wcef-settings' ) ); ?>" class="button button-primary">
						<?php esc_html_e( 'Configurer', 'wc-efacturation-france' ); ?>
					</a>
				</div>
			</div>
			<?php
		}
	}

	/**
	 * AJAX: Dismiss upgrade banner.
	 */
	public function ajax_dismiss_banner(): void {
		check_ajax_referer( 'wcef_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( 'Permission denied.' );
		}

		update_user_meta( get_current_user_id(), 'wcef_banner_dismissed', '1' );
		wp_send_json_success();
	}

	/**
	 * Plugin action links.
	 */
	public function action_links( array $links ): array {
		$custom = array(
			'<a href="' . esc_url( admin_url( 'admin.php?page=wcef-settings' ) ) . '">' . esc_html__( 'Reglages', 'wc-efacturation-france' ) . '</a>',
		);
		return array_merge( $custom, $links );
	}
}

// Activation/deactivation.
register_activation_hook( __FILE__, array( 'WCEF_Install', 'activate' ) );
register_deactivation_hook( __FILE__, array( 'WCEF_Install', 'deactivate' ) );

// Declare HPOS compatibility.
add_action( 'before_woocommerce_init', function () {
	if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
		\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
	}
} );

// Boot.
add_action( 'plugins_loaded', function () {
	if ( ! class_exists( 'WooCommerce' ) ) {
		add_action( 'admin_notices', function () {
			echo '<div class="notice notice-error"><p>' . esc_html__( 'E-Facturation France necessite WooCommerce.', 'wc-efacturation-france' ) . '</p></div>';
		} );
		return;
	}
	WC_EFacturation_France::instance();
} );
